<?php
/**
 * Installation process. Create tables, default options, etc.
 *
 * @category Core
 * @package  My Calendar
 * @author   Joe Dolson
 * @license  GPLv3
 * @link     https://www.joedolson.com/my-calendar/
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Get the original default list template. Any other template is considered customized.
 */
function mc_widget_default_list_template() {
	return '<strong>{timerange after=", "}{daterange}</strong> &#8211; {linking_title}';
}

/**
 * Default settings for widgets.
 *
 * @return array
 */
function mc_widget_defaults() {
	$default_template = mc_widget_default_list_template();

	$defaults = array(
		'upcoming' => array(
			'type'     => 'event',
			'before'   => 0,
			'after'    => 5,
			'template' => $default_template,
			'category' => '',
			'text'     => '',
			'title'    => 'Upcoming Events',
		),
		'today'    => array(
			'template' => $default_template,
			'category' => '',
			'title'    => 'Today\'s Events',
			'text'     => '',
		),
	);

	/**
	 * Customize the default values used for Upcoming and Today's events widgets.
	 *
	 * @hook mc_widget_defaults
	 *
	 * @param {array} $defaults Array of values used to set up upcoming and today's events lists in widgets.
	 *
	 * @return {array}
	 */
	return apply_filters( 'mc_widget_defaults', $defaults );
}

/**
 * Define variables to be saved in settings. (Formerly globals.)
 *
 * @param string $data Type of data to return.
 *
 * @return array
 */
function mc_globals( $data = 'all' ) {
	$grid_template = '
<span class="event-time value-title">{time}{endtime before="<span class=\'time-separator\'> - </span><span class=\'end-time\'>" after="</span>"}</span>
{image before="<div class=\'mc-event-image\'>" after="</div>"}
<div class="sub-details">
	{hcard before="<div class=\'mc-location\'>" after="</div>"}
	{excerpt before="<div class=\'mc-excerpt\'>" after="</div>"}
</div>';

	$single_template = '
<span class="event-time value-title" title="{dtstart}">{time}<span class="time-separator"> - </span><span class="end-time value-title" title="{dtend}">{endtime}</span></span>
{image before="<div class=\'mc-event-image\'>" after="</div>"}
<div class="event-data">
	{runtime before="<p class=\'mc-runtime\'>" after="</p>"}
	{categories before="<p class=\'mc-categories\'>" after="</p>"}
</div>
<div class="sub-details">
	{hcard before="<div class=\'mc-location\'>" after="</div>"}
	{description before="<div class=\'mc-description\'>" after="</div>"}
	{map before="<div class=\'mc-map\'>" after="</div>"}
</div>';

	$card_template = '
	<span class="event-time value-title">{time}{endtime before="<span class=\'time-separator\'> - </span><span class=\'end-time\'>" after="</span>"}</span>
	{image before="<div class=\'mc-event-image\'>" after="</div>"}
	<div class="sub-details">
		{excerpt before="<div class=\'mc-excerpt\'>" after="</div>"}
	</div>';

	if ( 'templates' === $data ) {
		$templates = array(
			'grid_template'   => addslashes( $grid_template ),
			'list_template'   => addslashes( $grid_template ),
			'mini_template'   => addslashes( $grid_template ),
			'single_template' => addslashes( $single_template ),
			'card_template'   => addslashes( $card_template ),
		);
		return $templates;
	}

	global $wpdb;
	$charset_collate  = $wpdb->get_charset_collate();
	$event_fifth_week = ( mc_no_fifth_week() ) ? 1 : 0;

	$initial_db = 'CREATE TABLE ' . my_calendar_table() . " (
 event_id INT(11) NOT NULL AUTO_INCREMENT,
 event_begin DATE NOT NULL,
 event_end DATE NOT NULL,
 event_title VARCHAR(255) NOT NULL,
 event_desc TEXT NOT NULL,
 event_short TEXT NOT NULL,
 event_registration TEXT NOT NULL,
 event_tickets VARCHAR(255) NOT NULL,
 event_time TIME,
 event_endtime TIME,
 event_recur CHAR(3),
 event_repeats TEXT,
 event_status INT(1) NOT NULL DEFAULT '1',
 event_author BIGINT(20) UNSIGNED,
 event_host BIGINT(20) UNSIGNED,
 event_category BIGINT(20) UNSIGNED NOT NULL DEFAULT '1',
 event_link TEXT,
 event_post BIGINT(20) UNSIGNED NOT NULL DEFAULT '0',
 event_link_expires TINYINT(1) NOT NULL,
 event_location BIGINT(20) UNSIGNED NOT NULL DEFAULT '0',
 event_label VARCHAR(255) NOT NULL,
 event_street VARCHAR(255) NOT NULL,
 event_street2 VARCHAR(255) NOT NULL,
 event_city VARCHAR(255) NOT NULL,
 event_state VARCHAR(255) NOT NULL,
 event_postcode VARCHAR(10) NOT NULL,
 event_region VARCHAR(255) NOT NULL,
 event_country VARCHAR(255) NOT NULL,
 event_url TEXT,
 event_longitude FLOAT(10,6) NOT NULL DEFAULT '0',
 event_latitude FLOAT(10,6) NOT NULL DEFAULT '0',
 event_zoom INT(2) NOT NULL DEFAULT '14',
 event_phone VARCHAR(32) NOT NULL,
 event_phone2 VARCHAR(32) NOT NULL,
 event_access TEXT,
 event_group_id INT(11) NOT NULL DEFAULT '0',
 event_span INT(1) NOT NULL DEFAULT '0',
 event_approved INT(1) NOT NULL DEFAULT '1',
 event_flagged INT(1) NOT NULL DEFAULT '0',
 event_hide_end INT(1) NOT NULL DEFAULT '0',
 event_holiday INT(1) NOT NULL DEFAULT '0',
 event_fifth_week INT(1) NOT NULL DEFAULT '$event_fifth_week',
 event_image TEXT,
 event_added TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
 PRIMARY KEY  (event_id),
 KEY event_category (event_category)
 ) $charset_collate;";

	$initial_occur_db = 'CREATE TABLE ' . my_calendar_event_table() . " (
 occur_id INT(11) NOT NULL AUTO_INCREMENT,
 occur_event_id INT(11) NOT NULL,
 occur_begin DATETIME NOT NULL,
 occur_end DATETIME NOT NULL,
 occur_group_id INT(11) NOT NULL DEFAULT '0',
 PRIMARY KEY  (occur_id),
 KEY occur_event_id (occur_event_id)
 ) $charset_collate;";

	$initial_cat_db = 'CREATE TABLE ' . my_calendar_categories_table() . " (
 category_id INT(11) NOT NULL AUTO_INCREMENT,
 category_name VARCHAR(255) NOT NULL,
 category_color VARCHAR(7) NOT NULL,
 category_icon VARCHAR(128) NOT NULL,
 category_private INT(1) NOT NULL DEFAULT '0',
 category_term INT(11) NOT NULL DEFAULT '0',
 PRIMARY KEY  (category_id)
 ) $charset_collate;";

	$initial_loc_rel_db = 'CREATE TABLE ' . my_calendar_location_relationships_table() . " (
 relationship_id INT(11) NOT NULL AUTO_INCREMENT,
 location_id INT(11) NOT NULL,
 post_id INT(11) NOT NULL DEFAULT '1',
 PRIMARY KEY  (relationship_id),
 KEY location_id (location_id)
 ) $charset_collate;";

	$initial_rel_db = 'CREATE TABLE ' . my_calendar_category_relationships_table() . " (
 relationship_id INT(11) NOT NULL AUTO_INCREMENT,
 event_id INT(11) NOT NULL,
 category_id INT(11) NOT NULL DEFAULT '1',
 PRIMARY KEY  (relationship_id),
 KEY event_id (event_id)
 ) $charset_collate;";

	$initial_loc_db = 'CREATE TABLE ' . my_calendar_locations_table() . " (
 location_id INT(11) NOT NULL AUTO_INCREMENT,
 location_label VARCHAR(255) NOT NULL,
 location_street VARCHAR(255) NOT NULL,
 location_street2 VARCHAR(255) NOT NULL,
 location_city VARCHAR(255) NOT NULL,
 location_state VARCHAR(255) NOT NULL,
 location_postcode VARCHAR(10) NOT NULL,
 location_region VARCHAR(255) NOT NULL,
 location_url TEXT,
 location_country VARCHAR(255) NOT NULL,
 location_longitude FLOAT(10,6) NOT NULL DEFAULT '0',
 location_latitude FLOAT(10,6) NOT NULL DEFAULT '0',
 location_zoom INT(2) NOT NULL DEFAULT '14',
 location_phone VARCHAR(32) NOT NULL,
 location_phone2 VARCHAR(32) NOT NULL,
 location_access TEXT,
 PRIMARY KEY  (location_id)
 ) $charset_collate;";

	$globals = array(
		'initial_db'         => $initial_db,
		'initial_occur_db'   => $initial_occur_db,
		'initial_rel_db'     => $initial_rel_db,
		'initial_loc_db'     => $initial_loc_db,
		'initial_loc_rel_db' => $initial_loc_rel_db,
		'initial_cat_db'     => $initial_cat_db,
		'grid_template'      => addslashes( $grid_template ),
		'list_template'      => addslashes( $grid_template ),
		'mini_template'      => addslashes( $grid_template ),
		'single_template'    => addslashes( $single_template ),
	);

	return $globals;
}

/**
 * Create demo content for new installs.
 */
function mc_create_demo_content() {
	global $wpdb;
	// If site has no categories, this is a new install.
	$categories = $wpdb->get_results( 'SELECT category_id FROM ' . my_calendar_categories_table() );  // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
	if ( empty( $categories ) ) {
		// Insert a category.
		mc_create_category(
			array(
				'category_name'  => 'General',
				'category_color' => '#fafafa',
				'category_icon'  => 'event.svg',
			)
		);
		// Insert a location.
		$add_event   = array(
			'location_label'     => 'Demo: Minnesota Orchestra',
			'location_street'    => '1111 Nicollet Mall',
			'location_street2'   => '',
			'location_city'      => 'Minneapolis',
			'location_state'     => 'MN',
			'location_postcode'  => '55403',
			'location_region'    => '',
			'location_country'   => 'United States',
			'location_url'       => 'https://www.minnesotaorchestra.org',
			'location_latitude'  => '44.9722',
			'location_longitude' => '-93.2749',
			'location_zoom'      => 16,
			'location_phone'     => '612-371-5600',
			'location_phone2'    => '',
			'location_access'    => '',
		);
		$results     = mc_insert_location( $add_event );
		$location_id = $results['location_id'];

		// Insert an event.
		$submit = array(
			// Begin strings.
			'event_begin'        => date( 'Y-m-d', strtotime( '+1 day' ) ), // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
			'event_end'          => date( 'Y-m-d', strtotime( '+1 day' ) ), // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
			'event_title'        => 'Demo: Florence Price: Symphony No. 3 in c minor',
			'event_desc'         => "<p>Florence Price's <a href='https://en.wikipedia.org/wiki/Symphony_No._3_(Price)'>Symphony No. 3</a> was commissioned by the Works Progress Administration's <a href='https://en.wikipedia.org/wiki/Federal_Music_Project'>Federal Music Project</a> during the height of the Great Depression. It was first performed at the Detroit Institute of Arts on November 6, 1940, by the Detroit Civic Orchestra under the conductor Valter Poole.</p><p>The composition is Price's third symphony, following her Symphony in E minor—the first symphony by a black woman to be performed by a major American orchestra—and her lost Symphony No. 2.</p>",
			'event_short'        => "Florence Price's Symphony No.3 was first performed on November 6th, 1940. It was Ms. Price's third symphony, following her lost Symphony No. 2",
			'event_time'         => '19:30:00',
			'event_endtime'      => '21:00:00',
			'event_link'         => 'https://www.youtube.com/watch?v=1jgJ1OkjnaI&list=OLAK5uy_lKldgbFTYBDa7WN6jf2ubB595wncDU7yc&index=2',
			'event_recur'        => 'S1',
			'event_image'        => plugins_url( '/images/demo/event.jpg', __FILE__ ),
			'event_access'       => '',
			'event_tickets'      => '',
			'event_registration' => '',
			'event_repeats'      => '',
			// Begin integers.
			'event_author'       => wp_get_current_user()->ID,
			'event_category'     => 1,
			'event_link_expires' => 0,
			'event_zoom'         => 16,
			'event_approved'     => 1,
			'event_host'         => wp_get_current_user()->ID,
			'event_flagged'      => 0,
			'event_fifth_week'   => 0,
			'event_holiday'      => 0,
			'event_group_id'     => 0,
			'event_span'         => 0,
			'event_hide_end'     => 0,
			// Array: removed before DB insertion.
			'event_categories'   => array( 1 ),
		);

		$event    = array( true, false, $submit, false, array() );
		$response = my_calendar_save( 'add', $event );
		$event_id = $response['event_id'];
		mc_update_event( 'event_location', (int) $location_id, $event_id );

		$e       = mc_get_first_event( $event_id );
		$post_id = $e->event_post;
		$image   = media_sideload_image( plugins_url( '/images/demo/event.jpg', __FILE__ ), $post_id, null, 'id' );

		if ( ! is_wp_error( $image ) ) {
			set_post_thumbnail( $post_id, $image );
		}
	}
}

/**
 * Get template values.
 */
function mc_template_settings() {
	$globals   = mc_globals( 'templates' );
	$templates = array(
		'title'      => '{time}: {title}',
		'title_list' => '{time}: {title}',
		'title_card' => '{title}',
		'title_solo' => '{title}',
		'link'       => '', // Empty because usage has a fallback value.
		'grid'       => $globals['grid_template'],
		'list'       => $globals['list_template'],
		'mini'       => $globals['mini_template'],
		'card'       => $globals['card_template'],
		'details'    => $globals['single_template'],
		'label'      => '', // Empty because usage has a fallback value.
	);

	return $templates;
}

/**
 * Fetch My Calendar's default settings array.
 *
 * @return array
 */
function mc_default_options() {
	$options = array(
		'display_single'               => array( 'ical', 'address', 'gcal', 'description', 'image', 'tickets', 'access', 'link', 'gmap_link' ),
		'display_main'                 => array( 'address', 'excerpt', 'image', 'tickets', 'access', 'gmap_link', 'more' ),
		'display_mini'                 => array( 'excerpt', 'image', 'more' ),
		'use_permalinks'               => 'true',
		'use_styles'                   => 'false',
		'show_months'                  => '1',
		'calendar_javascript'          => 'modal',
		'list_javascript'              => 'modal',
		'mini_javascript'              => 'modal',
		'ajax_javascript'              => '0',
		'show_js'                      => '',
		'notime_text'                  => '',
		'cancel_text'                  => '',
		'hide_icons'                   => 'true',
		'event_link_expires'           => 'false',
		'apply_color'                  => 'background',
		'input_options'                => mc_input_defaults(),
		'input_options_administrators' => 'false',
		'default_admin_view'           => 'list',
		'event_mail'                   => 'false',
		'event_mail_to'                => get_bloginfo( 'admin_email' ),
		'event_mail_from'              => get_bloginfo( 'admin_email' ),
		'event_mail_subject'           => '',
		'event_mail_message'           => '',
		'event_mail_bcc'               => '',
		'html_email'                   => 'true',
		'week_format'                  => '',
		'date_format'                  => '',
		'time_format'                  => '',
		'templates'                    => mc_template_settings(),
		'css_file'                     => '',
		'style_vars'                   => mc_style_variables(),
		'show_weekends'                => 'true',
		'convert'                      => 'true',
		'topnav'                       => 'toggle,timeframe,jump,nav',
		'bottomnav'                    => 'key,print',
		'default_direction'            => 'DESC',
		'remote'                       => 'false',
		'gmap_api_key'                 => '',
		'uri_id'                       => '',
		'open_uri'                     => '',
		'use_permalinks'               => 'true',
		'drop_tables'                  => '',
		'drop_settings'                => '',
		'api_enabled'                  => '',
		'api_key'                      => '',
		'default_sort'                 => '',
		'current_table'                => '',
		'open_day_uri'                 => 'false',
		'mini_marker'                  => 'events',
		'upcoming_events_navigation'   => 'false',
		'mini_uri'                     => '',
		'show_list_info'               => '',
		'show_list_events'             => '',
		'event_title_template'         => '',
		'heading_text'                 => '',
		'notime_text'                  => '',
		'cancel_text'                  => '',
		'hosted_by'                    => '',
		'posted_by'                    => '',
		'buy_tickets'                  => '',
		'event_accessibility'          => '',
		'view_full'                    => '',
		'week_caption'                 => '',
		'next_events'                  => '',
		'previous_events'              => '',
		'today_events'                 => '',
		'caption'                      => '',
		'subscribe'                    => '',
		'export'                       => '',
		'month_format'                 => '',
		'time_format'                  => '',
		'location_controls'            => '',
		'cpt_base'                     => 'mc-events',
		'location_cpt_base'            => 'mc-locations',
		'uri_query'                    => '',
		'default_category'             => '',
		'default_access_terms'         => array(),
		'default_location_terms'       => array(),
		'skip_holidays_category'       => '',
		'hide_icons'                   => '',
		'use_list_template'            => '',
		'use_card_template'            => '',
		'use_mini_template'            => '',
		'use_details_template'         => '',
		'use_grid_template'            => '',
		'migrated'                     => '',
		'list_link_titles'             => 'true',
		'default_location'             => '',
		'hide_past_dates'              => 'false',
		'map_service'                  => 'google',
		'disable_legacy_templates'     => 'false',
		'maptype'                      => 'roadmap',
		'views'                        => array( 'calendar', 'list', 'mini' ),
		'time_views'                   => array( 'month', 'week', 'day' ),
		'list_template'                => '',
	);

	/**
	 * Filter the default values for My Calendar settings.
	 *
	 * @hook mc_default_options
	 *
	 * @param {array} $options Array of My Calendar settings.
	 *
	 * @return {array}
	 */
	return apply_filters( 'mc_default_options', $options );
}

/**
 * Do initial plugin installation.
 */
function mc_initial_install() {
	delete_option( 'mc_uninstalled' );
	$globals = mc_globals();
	$options = mc_default_options();
	add_option( 'my_calendar_options', $options );
	add_site_option( 'mc_multisite', '0' );

	mc_add_roles();
	$has_uri = mc_locate_calendar();
	if ( false === $has_uri['response'] ) {
		// if mc_locate_calendar returns a string, no valid URI was found.
		$slug = sanitize_title( __( 'My Calendar', 'my-calendar' ) );
		mc_generate_calendar_page( $slug );
	}
	require_once ABSPATH . 'wp-admin/includes/upgrade.php';
	dbDelta( $globals['initial_db'] );
	dbDelta( $globals['initial_occur_db'] );
	dbDelta( $globals['initial_cat_db'] );
	dbDelta( $globals['initial_rel_db'] );
	dbDelta( $globals['initial_loc_db'] );
	dbDelta( $globals['initial_loc_rel_db'] );
}

/**
 * Create new calendar page
 *
 * @param string $slug Intended page name.
 *
 * @return int $post_ID for new page or previously existing page with same name.
 */
function mc_generate_calendar_page( $slug ) {
	global $current_user;
	$current_user = wp_get_current_user();
	$page_by_path = get_page_by_path( $slug );
	$page_status  = ( $page_by_path ) ? $page_by_path->post_status : '';
	$allowed      = array( 'private', 'publish' );
	if ( ! $page_by_path || ( $page_by_path && ! in_array( $page_status, $allowed, true ) ) ) {
		$page      = array(
			'post_title'   => __( 'My Calendar', 'my-calendar' ),
			'post_status'  => 'publish',
			'post_type'    => 'page',
			'post_author'  => $current_user->ID,
			'ping_status'  => 'closed',
			'post_content' => '<!-- wp:shortcode -->[my_calendar id="my-calendar"]<!-- /wp:shortcode -->',
		);
		$post_ID   = wp_insert_post( $page );
		$post_slug = wp_unique_post_slug( $slug, $post_ID, 'publish', 'page', 0 );
		wp_update_post(
			array(
				'ID'        => $post_ID,
				'post_name' => $post_slug,
			)
		);
	} else {
		// If the path exists, and is published, assign.
		$post_ID = $page_by_path->ID;
	}
	mc_update_option( 'uri_id', $post_ID );

	return $post_ID;
}

/**
 * See whether there are importable calendars present.
 */
function mc_check_imports() {
	if ( 'true' !== get_option( 'ko_calendar_imported' ) ) {
		if ( function_exists( 'calendar_check' ) ) {
			wp_admin_notice(
				// translators: link to migration screen.
				sprintf( __( 'My Calendar has identified that you have the Calendar plugin. You can import those events and categories into the My Calendar database. <a href="%s">Would you like to import these events?</a>', 'my-calendar' ), admin_url( 'admin.php?page=my-calendar-migrate' ) ),
				array(
					'type' => 'info',
				)
			);
		}
	}
}

/**
 * Transition location relationships into own table.
 *
 * @param int $location_id Location ID from location table.
 * @param int $location_post Post ID from posts table.
 *
 * @since 3.3.0
 */
function mc_transition_location( $location_id, $location_post ) {
	global $wpdb;

	if ( $location_post ) {
		$wpdb->insert(
			my_calendar_location_relationships_table(),
			array(
				'location_id' => $location_id,
				'post_id'     => $location_post,
			),
			array( '%d', '%d' )
		);
		delete_post_meta( $location_post, '_mc_location_id' );
	}
}

/**
 * Migrate event accessibility from database to taxonomy.
 */
function mc_migrate_event_accessibility() {
	if ( 'true' !== get_option( 'mc_event_access_migration_completed' ) ) {
		$options = mc_event_access();
		// Add terms.
		foreach ( $options as $value ) {
			wp_insert_term( $value, 'mc-event-access' );
		}
		mc_migrate_event_access();
	}
}

/**
 * Migrate Event access data.
 *
 * @param int $limit Number of events to import.
 */
function mc_migrate_event_access( $limit = 200 ) {
	// Get selection of events not already migrated.
	$events = get_posts(
		array(
			'post_type'      => 'mc-events',
			'meta_query'     => array(
				array(
					'key'     => '_mc_event_access',
					'value'   => '',
					'compare' => '!=',
				),
				array(
					'key'     => '_mc_access_migrated',
					'compare' => 'NOT EXISTS',
				),
			),
			'posts_per_page' => $limit,
			'fields'         => 'ids',
		)
	);
	$count  = count( $events );
	if ( 0 === $count ) {
		// No longer need this key.
		delete_post_meta_by_key( '_mc_access_migrated' );
		update_option( 'mc_event_access_migration_completed', 'true', 'no' );
	} else {
		// Iterate events and save meta data as taxonomy data.
		foreach ( $events as $event ) {
			$access = get_post_meta( $event, '_mc_event_access', true );
			if ( is_array( $access ) ) {
				$terms = array();
				$notes = isset( $access['notes'] ) ? $access['notes'] : '';
				if ( $notes ) {
					$notes = sanitize_textarea_field( $notes );
					update_post_meta( $event, '_mc_event_access', $notes );
				} else {
					delete_post_meta( $event, '_mc_event_access' );
				}
				unset( $access['notes'] );
				foreach ( $access as $type ) {
					$terms[] = $type;
				}
				wp_set_object_terms( $event, $terms, 'mc-event-access' );
				update_post_meta( $event, '_mc_access_migrated', 'true' );
			}
		}

		if ( false === as_has_scheduled_action( 'mc_event_access_migration' ) ) {
			as_schedule_recurring_action( strtotime( '+1 minutes' ), 60, 'mc_event_access_migration', array(), 'my-calendar' );
		}
	}
}
// Register this action in action scheduler.
add_action( 'mc_event_access_migration', 'mc_migrate_event_access' );

/**
 * Suspend migration process if completed.
 */
function mc_check_migration_progress() {
	if ( 'true' === get_option( 'mc_event_access_migration_completed' ) ) {
		as_unschedule_all_actions( 'mc_event_access_migration' );
	}
	if ( 'true' === get_option( 'mc_location_access_migration_completed' ) ) {
		as_unschedule_all_actions( 'mc_location_access_migration' );
	}
}
add_action( 'admin_init', 'mc_check_migration_progress' );

/**
 * Migrate location accessibility from database to taxonomy.
 */
function mc_migrate_location_accessibility() {
	if ( 'true' !== get_option( 'mc_location_access_migration_completed' ) ) {
		$options = mc_location_access();
		// Add terms.
		foreach ( $options as $value ) {
			wp_insert_term( $value, 'mc-location-access' );
		}
		mc_migrate_location_access();
	}
}

/**
 * Migrate Location access data.
 *
 * @param int $limit Number of events to import.
 */
function mc_migrate_location_access( $limit = 200 ) {
	global $wpdb;
	// Get all locations with a value saved for accessibility.
	$locations = $wpdb->get_results( $wpdb->prepare( 'SELECT location_id, location_access FROM ' . my_calendar_locations_table() . ' WHERE location_access != "" LIMIT 0,%d', $limit ) ); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
	// Get location access terms.
	$options = mc_location_access();
	// Get selection of events not already migrated.
	$count = count( $locations );
	if ( 0 === $count ) {
		// No longer need this key.
		delete_post_meta_by_key( '_mc_access_migrated' );
		update_option( 'mc_location_access_migration_completed', 'true', 'no' );
	} else {
		// Iterate locations and save meta data as taxonomy data.
		foreach ( $locations as $location ) {
			$access  = ( $location->location_access ) ? unserialize( $location->location_access ) : array();
			$post_id = mc_get_location_post( $location->location_id, false );
			if ( is_array( $access ) ) {
				$terms = array();
				foreach ( $access as $type ) {
					$value   = ( is_numeric( $type ) ) ? $options[ $type ] : $type;
					$terms[] = $value;
				}

				wp_set_object_terms( $post_id, $terms, 'mc-location-access' );
			}
			// remove location access data.
			mc_update_location_field( 'location_access', '', $location->location_id );
		}

		if ( false === as_has_scheduled_action( 'mc_location_access_migration' ) ) {
			as_schedule_recurring_action( strtotime( '+1 minutes' ), 60, 'mc_location_access_migration', array(), 'my-calendar' );
		}
	}
}
// Register this action in action scheduler.
add_action( 'mc_location_access_migration', 'mc_location_event_access' );
