<?php
/**
 * Handle My Calendar submissions via AJAX.
 *
 * @category Submissions
 * @package  My Calendar Pro
 * @author   Joe Dolson
 * @license  GPLv3
 * @link     https://www.joedolson.com/my-calendar-pro/
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Enqueue AJAX submission scripts.
 */
function mcs_submit_ajax_scripts() {
	if ( ! is_admin() && mcs_user_can_submit_events() ) {
		global $mcs_version;
		$submit_url = plugins_url( 'js/jquery.mcs-submit.min.js', __FILE__ );
		if ( SCRIPT_DEBUG && true === SCRIPT_DEBUG ) {
			$mcs_version = $mcs_version . uniqid();
			$submit_url  = plugins_url( 'js/jquery.mcs-submit.js', __FILE__ );
		}
		wp_enqueue_script( 'mcs.submit', $submit_url, array( 'jquery' ), $mcs_version, true );
		wp_localize_script(
			'mcs.submit',
			'mcs',
			array(
				'ajaxurl'          => admin_url( 'admin-ajax.php' ),
				'security'         => wp_create_nonce( 'mcs-submit-ajax' ),
				'action'           => 'mcs_submit_ajax',
				'required'         => __( 'Required field: ', 'my-calendar-pro' ),
				'requiredCheckbox' => __( 'Error: Select at least one option.', 'my-calendar-pro' ),
				'requiredRadio'    => __( 'Error: Select an option.', 'my-calendar-pro' ),
				'heading'          => __( 'Required fields are not completed', 'my-calendar-pro' ),
			)
		);
	}
}
add_action( 'wp_enqueue_scripts', 'mcs_submit_ajax_scripts' );

/**
 * Handle Form submission.
 */
function mcs_submit_ajax() {
	if ( isset( $_REQUEST['action'] ) && 'mcs_submit_ajax' === $_REQUEST['action'] ) {
		$post = $_REQUEST['post']; // Post data is parsed before sanitizing.
		// Parse serialized data.
		parse_str( $post, $data );
		$data = map_deep( $data, 'mcs_ajax_sanitize' );

		$security = $_REQUEST['security'];
		if ( ! wp_verify_nonce( $security, 'mcs-submit-ajax' ) ) {
			wp_send_json(
				array(
					'success'  => 0,
					'response' => 'Invalid Nonce',
				)
			);
		} else {
			$response = mcs_processor( $data );
			if ( ! $response ) {
				wp_send_json(
					array(
						'success'  => 0,
						'response' => 'My Calendar threw an unidentified submission error.',
					)
				);
			} else {
				$message = $response[0];
				$data    = $response[1];
				$success = $response[2];

				wp_send_json(
					array(
						'success'  => $success,
						'response' => $message,
						'data'     => $data,
					)
				);
			}
		}
	}
}
add_action( 'wp_ajax_mcs_submit_ajax', 'mcs_submit_ajax' );
add_action( 'wp_ajax_nopriv_mcs_submit_ajax', 'mcs_submit_ajax' );

/**
 * Handle AJAX upload image file.
 */
function mcs_ajax_upload_file() {
	check_ajax_referer( 'mcs-submit-ajax', 'nonce' );

	$wp_upload_dir = wp_upload_dir();
	$file_path     = trailingslashit( $wp_upload_dir['path'] ) . sanitize_file_name( $_POST['file'] );
	$file_data     = mcs_decode_chunk( $_POST['file_data'] );
	$post_id       = absint( $_POST['event_post_id'] );
	$file_alt      = sanitize_text_field( $_POST['file_alt'] );
	$complete      = ( 'true' === $_POST['complete'] ) ? true : false;
	if ( false === $file_data ) {
		wp_send_json_error();
	}

	$saved = file_put_contents( $file_path, $file_data, FILE_APPEND );
	if ( false !== $saved ) {
		if ( $complete ) {
			require_once ABSPATH . 'wp-admin/includes/media.php';
			require_once ABSPATH . 'wp-admin/includes/file.php';
			require_once ABSPATH . 'wp-admin/includes/image.php';
			$media_url = trailingslashit( $wp_upload_dir['url'] ) . sanitize_file_name( $_POST['file'] );
			// $desc parameter becomes the post title, used as alt attribute in media library admin.
			$desc          = $file_alt;
			$attachment_id = media_sideload_image( $media_url, $post_id, $desc, 'id' );
			if ( is_int( $attachment_id ) ) {
				update_post_meta( $attachment_id, '_wp_attachment_image_alt', $file_alt );
				$data = array(
					'success'       => true,
					'attachment_id' => $attachment_id,
					'src'           => wp_get_attachment_image_url( $attachment_id, 'medium' ),
				);
				wp_send_json( $data );
			} else {
				$data = array(
					'success'       => false,
					'attachment_id' => false,
					'errors'        => $attachment_id, // WP Error object.
				);
				wp_send_json( $data );
			}
		} else {
			// A chunk was saved, but upload is not finished.
			$data = array(
				'success'       => true,
				'attachment_id' => false,
			);
			wp_send_json( $data );
		}
	}
	$data = array(
		'success'       => false,
		'attachment_id' => false,
	);
	wp_send_json( $data );
}
add_action( 'wp_ajax_mcs_upload_file', 'mcs_ajax_upload_file' );
add_action( 'wp_ajax_nopriv_mcs_upload_file', 'mcs_ajax_upload_file' );

/**
 * Decode a base64 encoded file chunk.
 *
 * @param string $data base64 chunk.
 *
 * @return string
 */
function mcs_decode_chunk( $data ) {
	$data = explode( ';base64,', $data );

	if ( ! is_array( $data ) || ! isset( $data[1] ) ) {
		return false;
	}

	$data = base64_decode( $data[1] );
	if ( ! $data ) {
		return false;
	}

	return $data;
}

/**
 * Sanitize data sent to update pr submit a new event.
 *
 * @param string $content String to sanitize.
 *
 * @return string
 */
function mcs_ajax_sanitize( $content ) {
	return wp_kses_post( $content );
}
