<?php
/**
 * Licensing
 *
 * @category Licensing
 * @package  My Calendar Pro
 * @author   Joe Dolson
 * @license  GPLv3
 * @link     https://www.joedolson.com/my-calendar-pro/
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}


/**
 * Recheck license status.
 *
 * @param string $key license key.
 *
 * @return void|boolean
 */
function mcs_recheck_license( $key ) {
	$transient = get_transient( 'mcs_last_check' );
	if ( $transient ) {
		return;
	} else {
		$api_params = array(
			'edd_action' => 'check_license',
			'license'    => $key,
			'item_id'    => 5734, // the ID of our product in EDD.
			'url'        => home_url(),
		);

		// Call the custom API.
		$response = wp_remote_post(
			EDD_MCP_STORE_URL,
			array(
				'timeout' => 15,
				'body'    => $api_params,
			)
		);
		// make sure the response came back okay.
		if ( is_wp_error( $response ) ) {
			return false;
		}
		// decode the license data.
		$license_data = json_decode( wp_remote_retrieve_body( $response ) );
		update_option( 'mcs_license_status', array( current_time( 'timestamp' ) => $license_data ) ); // phpcs:ignore WordPress.DateTime.CurrentTimeTimestamp.Requested
		// Don't check more than once a day.
		set_transient( 'mcs_last_check', true, DAY_IN_SECONDS );
	}
}

/**
 * Check the license.
 *
 * @param array $post POST data.
 *
 * @return string
 */
function mcs_check_license( $post ) {
	// listen for our activate button to be clicked.
	if ( isset( $post['mcs_license_key'] ) ) {
		// run a quick security check.
		if ( ! check_admin_referer( 'license', '_wpnonce' ) ) {
			return; // get out if we didn't click the Activate button.
		}
		// retrieve the license from the database.
		$license = trim( sanitize_text_field( $post['mcs_license_key'] ) );
		// data to send in our API request.
		$api_params = array(
			'edd_action' => 'activate_license',
			'license'    => $license,
			'item_id'    => 5734, // the name of our product in EDD.
			'url'        => home_url(),
		);

		// Call the custom API.
		$response = wp_remote_post(
			EDD_MCP_STORE_URL,
			array(
				'timeout' => 15,
				'body'    => $api_params,
			)
		);

		// make sure the response came back okay.
		if ( is_wp_error( $response ) ) {
			return false;
		}
		// decode the license data.
		$license_data = json_decode( wp_remote_retrieve_body( $response ) );
		update_option( 'mcs_license_status', array( current_time( 'timestamp' ) => $license_data ) ); // phpcs:ignore WordPress.DateTime.CurrentTimeTimestamp.Requested

		return ( 'valid' === $license_data->license ) ? 'valid' : $license_data->error;
	}
}

/**
 * Verify license key pattern.
 *
 * @param array $post POST data.
 *
 * @return string
 */
function mcs_verify_key( $post ) {
	$key = isset( $post['mcs_license_key'] ) ? sanitize_text_field( $post['mcs_license_key'] ) : false;
	// If values are filled with asterisks, do not update; these are masked values.
	if ( stripos( $key, '***' ) !== false ) {
		$key = false;
	}

	if ( $key ) {
		delete_option( 'mcs_license_status' );
		$confirmation = mcs_check_license( $post );
	} else {
		$confirmation = 'deleted';
	}
	update_option( 'mcs_license_key', $key );
	update_option( 'mcs_license_key_valid', $confirmation );
	if ( 'valid' === $confirmation ) {
		$message = __( 'Your My Calendar Pro license key has been activated! Enjoy!', 'my-calendar-pro' );
	} elseif ( 'deleted' === $confirmation ) {
		$message = __( 'You have deleted your My Calendar Pro license key.', 'my-calendar-pro' );
	} elseif ( in_array( $confirmation, array( 'missing', 'disabled', 'no_activations_left', 'expired', 'key_mismatch', 'invalid' ), true ) ) {
		switch ( $confirmation ) {
			case 'missing':
				$message = __( 'No license key was passed to the license server.', 'my-calendar-pro' );
				break;
			case 'disabled':
				$message = __( 'This license key has been revoked.', 'my-calendar-pro' );
				break;
			case 'no_activations_left':
				$message = __( 'The maximum number of license activations has been reached.', 'my-calendar-pro' );
				break;
			case 'expired':
				// translators: Renewal URL.
				$message = sprintf( __( 'Your license has expired. <a href="%s">Renew your license now!</a>', 'my-calendar-pro' ), "https://www.joedolson.com/checkout/?edd_license_key=$key&download_id=5734" );
				break;
			case 'key_mismatch':
				$message = __( 'That is a valid license key, but for a different product.', 'my-calendar-pro' );
				break;
			case 'invalid':
				$message = sprintf( __( 'That license key is not valid.', 'my-calendar-pro' ), "https://www.joedolson.com/checkout/?edd_license_key=$key&download_id=5734" );
				break;
		}
		// Translators: Account URL.
		$message .= ( 'expired' !== $confirmation ) ? ' ' . sprintf( __( '<a href="%s">Log in to your account to check</a>', 'my-calendar-pro' ), 'https://www.joedolson.com/account/' ) : '';
	} else {
		$message = __( 'My Calendar Pro received an unexpected message from the license server. Please try again!', 'my-calendar-pro' );
	}
	$message = ( '' !== $message ) ? " $message " : $message; // just add a space.

	return $message;
}

add_filter( 'mcs_custom_settings_update', 'mcs_activate_license', 10, 2 );
/**
 * Activate license.
 *
 * @param string $value HTML to display in settings. Default false.
 * @param array  $post POST array.
 *
 * @return string
 */
function mcs_activate_license( $value, $post ) {
	// save settings.
	// If values are filled with asterisks, do not update; these are masked values.
	if ( isset( $post['mcs_license_key'] ) ) {
		if ( stripos( $post['mcs_license_key'], '***' ) !== false ) {
			return __( 'A concealed license key was ignored. Re-enter your key to update.', 'my-calendar-pro' );
		}
		if ( get_option( 'mcs_license_key' ) !== $post['mcs_license_key'] ) {
			$verify = mcs_verify_key( $post );
		} else {
			$verify = '';
		}
		if ( '' !== trim( $verify ) ) {
			return $verify;
		}
	}

	return $value;
}

add_filter( 'mcs_settings_tabs', 'mcs_license_tabs' );
/**
 * Set up licensing tab.
 *
 * @param array $tabs all tabs.
 *
 * @return array tabs
 */
function mcs_license_tabs( $tabs ) {
	$tabs['license'] = __( 'License', 'my-calendar-pro' );

	return $tabs;
}

add_filter( 'mcs_settings_panels', 'mcs_license_key' );
/**
 * Set up licensing panel.
 *
 * @param array $panels All panels.
 *
 * @return array
 */
function mcs_license_key( $panels ) {
	$mcs_license_key = ( ! defined( 'MCS_LICENSE_KEY' ) ) ? get_option( 'mcs_license_key' ) : MCS_LICENSE_KEY;
	mcs_recheck_license( $mcs_license_key );

	$mcs_license_key_valid = ( '' !== get_option( 'mcs_license_key_valid', '' ) ) ? ' (' . get_option( 'mcs_license_key_valid' ) . ')' : '';
	$mcs_license           = get_option( 'mcs_license_status' );
	$output                = '';
	$expired               = '';
	$valid                 = false;

	if ( empty( $mcs_license_key ) ) {
		$output .= "<div class='notice notice-error inline'><p>" . __( 'No license set. Add your license key for updates and support.', 'my-calendar-pro' ) . '</p></div>';
	} elseif ( ! empty( $mcs_license ) ) {
		$type = 'info';
		foreach ( $mcs_license as $test => $status ) {
			if ( ! is_object( $status ) ) {
				continue;
			}
			$test    = ( ! is_numeric( $test ) ) ? time() : $test;
			$license = $status->license;
			if ( 'valid' === $license ) {
				$valid   = true;
				$type    = 'success';
				$date    = date_i18n( 'Y-m-d', $test );
				$expire  = ( property_exists( $status, 'expires' ) ) ? $status->expires : false;
				$expires = ( 'lifetime' === $expire ) ? 'Lifetime' : date_i18n( 'Y-m-d', strtotime( $expire ) );
				if ( ! $expire ) {
					$expired = '<p>' . esc_html__( 'License check failed. Please request support at plugins@joedolson.com', 'my-calendar-pro' ) . '</p>';
				} elseif ( 'lifetime' !== $status->expires && strtotime( $status->expires ) < $test ) {
					// Translators: Link to renew license.
					$expired = '<p>' . sprintf( __( 'Your license has expired. <a href="%s">Renew your license now!</a>', 'my-calendar-pro' ), "https://www.joedolson.com/checkout/?edd_license_key=$mcs_license_key&download_id=5734" ) . '</p>';
				}
				$email = ( property_exists( $status, 'customer_email' ) ) ? $status->customer_email : '';
				if ( $expire && 'Lifetime' === $expires ) {
					// Translators: 1) type of license, 2) customer email.
					$license_check = sprintf( __( 'You have a %1$s license. Thank you! Your customer email is %2$s.', 'my-calendar-pro' ), "<strong>$expires</strong>", "<strong>$email</strong>" );
				} else {
					// Translators: 1) Date license expires, 2) current license status, 3) Customer email.
					$license_check = sprintf( __( 'License expires on %1$s. License is %2$s. Your customer email is %3$s.', 'my-calendar-pro' ), "<strong>$expires</strong>", "<strong>$license</strong>", "<strong>$email</strong>" );
				}
				// Translators: Date of last license verification.
				$output .= sprintf( __( 'Last Checked: %s', 'my-calendar-pro' ), "<strong>$date</strong>" ) . "</p><p class='mcs-check-license'>" . $license_check . '</p>';
			} else {
				$valid = false;
				$type  = 'error';
				if ( property_exists( $status, 'error' ) && 'disabled' === $status->error ) {
					// Translators: Email address for support.
					$output .= sprintf( __( 'That license key has been disabled. If you believe this is in error, please submit a support request to %s', 'my-calendar-pro' ), '<a href="mailto:plugins@joedolson.com">plugins@joedolson.com</a>' );
				} elseif ( property_exists( $status, 'error' ) && 'expired' === $status->error ) {
					$renew_url = esc_url( "https://www.joedolson.com/checkout/?edd_license_key=$license&download_id=5734" );
					// Translators: renewal URL.
					$output .= sprintf( __( 'That license key has expired. Please <a href="%s">renew your license key</a>!', 'my-calendar-pro' ), $renew_url );
				} else {
					$output .= __( 'That license key is not a valid key. Please double-check the key and try again!', 'my-calendar-pro' );
				}
			}
			$output = "<div class='notice notice-$type inline'>" . wpautop( $output ) . wpautop( $expired ) . '</div>';
		}
	}

	$network_activated = is_multisite() && is_main_site() && is_plugin_active_for_network( 'my-calendar/my-calendar.php' ) ? true : false;
	// Only render licensing section on main site in networks when network activated.
	if ( $network_activated || ( is_multisite() && ! $network_activated ) || ! is_multisite() ) {
		$panels['license']['content'] = '
			<h2>' . __( 'Activate Your License Key', 'my-calendar-pro' ) . '</h2>
			<div class="inside">
				' . $output . '
				<p class="license">
				<label for="mcs_license_key">' . __( 'License key', 'my-calendar-pro' ) . $mcs_license_key_valid . '</label><br /><input type="text" name="mcs_license_key" id="mcs_license_key" class="widefat" size="60" value="' . esc_attr( trim( mcs_mask_attr( $mcs_license_key, $valid ) ) ) . '" />
				</p>
				{submit}
			</div>';
	} else {
		$panels['license']['content'] = '
			<h2>' . __( 'License Key Status', 'my-calendar-pro' ) . '</h2>
			<div class="inside">
				' . $output . '
			</div>';
	}

	$label = ( ' (valid)' === $mcs_license_key_valid ) ? __( 'Update License Key', 'my-calendar-pro' ) : __( 'Activate License', 'my-calendar-pro' );

	$panels['license']['label'] = $label;

	return $panels;
}

/**
 * Mask secure values. If license key is valid, conceal it.
 *
 * @param string $value Current license key from DB.
 * @param bool   $valid Is this key valid.
 *
 * @return string
 */
function mcs_mask_attr( $value, $valid ) {
	if ( ! $valid ) {
		return $value;
	}
	$count  = strlen( $value );
	$substr = substr( $value, -5 );
	$return = str_pad( $substr, $count, '*', STR_PAD_LEFT );

	return $return;
}
