<?php
/**
 * Manage API for sharing events between calendars.
 *
 * @category Features
 * @package  My Calendar Pro
 * @author   Joe Dolson
 * @license  GPLv3
 * @link     https://www.joedolson.com/my-calendar-pro/
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

add_filter( 'mcs_custom_settings_update', 'mcs_api_update', 10, 2 );
/**
 * Update API settings.
 *
 * @param boolean $value Is updated.
 * @param array   $post POST data.
 *
 * @return string update info.
 */
function mcs_api_update( $value, $post ) {
	// save settings.
	$message = '';
	if ( isset( $post['mcs_api'] ) ) {
		$setting = ( isset( $post['mcs_api'] ) ) ? sanitize_title( $post['mcs_api'] ) : 'disabled';
		update_option( 'mcs_api', $setting );

		switch ( $setting ) {
			case 'disabled':
				$message = __( 'The REST API is disabled.', 'my-calendar-pro' );
				break;
			case 'server-api':
				// Translators: URL to manage client sites.
				$message = sprintf( __( 'This website has been configured as the REST API server for My Calendar. <a href="%s">Manage API Clients</a>', 'my-calendar-pro' ), admin_url( 'admin.php?page-my-calendar-submissions-server' ) );
				break;
			case 'client-api':
				// Translators: URL to manage Endpoints.
				$message = sprintf( __( 'This website has been configured as a REST API client for My Calendar. <a href="%s">Manage API Endpoints</a>', 'my-calendar-pro' ), admin_url( 'admin.php?page-my-calendar-submissions-client' ) );
				break;
		}

		return $message;
	}

	return $value;
}

add_filter( 'mcs_settings_tabs', 'mcs_api_tabs' );
/**
 * Set up settings tab for REST API.
 *
 * @param array $tabs All My Calendar pro tabs.
 *
 * @return array updated tabs.
 */
function mcs_api_tabs( $tabs ) {
	$tabs['api'] = __( 'REST API', 'my-calendar-pro' );

	return $tabs;
}

add_filter( 'mcs_settings_panels', 'mcs_api_settings' );
/**
 * Set up settings for API config..
 *
 * @param array $panels All My Calendar pro panels.
 *
 * @return array updated panels.
 */
function mcs_api_settings( $panels ) {
	$mcs_api    = apply_filters( 'mcs_api_state', get_option( 'mcs_api', 'disabled' ) );
	$serverlink = '';
	$clientlink = '';
	$linked     = ( 'client-api' === $mcs_api ) ? admin_url( 'admin.php?page=my-calendar-submissions-client' ) : admin_url( 'admin.php?page=my-calendar-submissions-server' );
	$linktext   = ( 'client-api' === $mcs_api ) ? __( 'Endpoints', 'my-calendar-pro' ) : __( 'Clients', 'my-calendar-pro' );

	if ( 'client-api' === $mcs_api ) {
		// Translators: Link URL, client or server.
		$clientlink = ' ' . sprintf( __( '<a href="%1$s">Configure %2$s</a>', 'my-calendar-pro' ), esc_url( $linked ), $linktext );
	} elseif ( 'server-api' === $mcs_api ) {
		// Translators: Link URL, client or server.
		$serverlink = ' ' . sprintf( __( '<a href="%1$s">Configure %2$s</a>', 'my-calendar-pro' ), esc_url( $linked ), $linktext );
	}
	$current_status = '';
	if ( $mcs_api && 'disabled' !== $mcs_api ) {
		$current_status = ( 'client-api' === $mcs_api ) ? __( 'This site is enabled as a API client, and can post events to enabled remote calendars.', 'my-calendar-pro' ) : __( 'This site is enabled as an API server, and can receive events from enabled remote clients.', 'my-calendar-pro' );
		$current_status = '<p><em>' . $current_status . '</em></p>';
	}

	$panels['api'] = '
		<h2>' . __( 'Configure REST API', 'my-calendar-pro' ) . '</h2>
		<div class="inside">
			<p>' . __( 'Your site can be an <strong>API client</strong> (post to another calendar if that site is an API server), or an <strong>API server</strong> (receive events posted from sites enabled as API clients.)', 'my-calendar-pro' ) . '</p>
			' . $current_status . '
			<ul>
				<li><input type="radio" name="mcs_api" id="mcs_api_disabled" value="disabled" ' . checked( $mcs_api, 'disabled', false ) . '/> <label for="mcs_api_disabled">' . __( 'REST API features disabled', 'my-calendar-pro' ) . '</label></li>
				<li><input type="radio" name="mcs_api" id="mcs_api_server" value="server-api" ' . checked( $mcs_api, 'server-api', false ) . '/> <label for="mcs_api_server">' . __( 'This site is an API server. (REST API enabled)', 'my-calendar-pro' ) . $serverlink . '</label></li>
				<li><input type="radio" name="mcs_api" id="mcs_api_client" value="client-api" ' . checked( $mcs_api, 'client-api', false ) . '/> <label for="mcs_api_client">' . __( 'This site is an API client. (REST API disabled)', 'my-calendar-pro' ) . $clientlink . '</label></li>
			</ul>
			{submit}
		</div>';

	return $panels;
}
