<?php
/**
 * Geolocation functions.
 *
 * @category Features
 * @package  My Calendar Pro
 * @author   Joe Dolson
 * @license  GPLv3
 * @link     https://www.joedolson.com/my-calendar-pro/
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use GeoIp2\Database\Reader;

/**
 * Get user location by IP.
 */
function mc_get_user_ip() {
	$client  = isset( $_SERVER['HTTP_CLIENT_IP'] ) ? sanitize_text_field( $_SERVER['HTTP_CLIENT_IP'] ) : '';
	$forward = isset( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ? sanitize_text_field( $_SERVER['HTTP_X_FORWARDED_FOR'] ) : '';
	$remote  = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( $_SERVER['REMOTE_ADDR'] ) : '';

	if ( filter_var( $client, FILTER_VALIDATE_IP ) ) {
		$ip = $client;
	} elseif ( filter_var( $forward, FILTER_VALIDATE_IP ) ) {
		$ip = $forward;
	} else {
		$ip = $remote;
	}

	return $ip;
}

/**
 * Submit a query to recenter geolocation map.
 *
 * @param array $address Address information.
 *
 * @return string
 */
function mcs_geolocate_form( $address = array() ) {
	$nonce  = wp_create_nonce( 'mc-geolocate-nonce' );
	$fields = array(
		'street' => array(
			'label' => __( 'Street Address', 'my-calendar-pro' ),
			'auto'  => 'address-line1',
		),
	);
	$fields = apply_filters( 'mc_geolocate_fields', $fields );
	if ( empty( $fields ) ) {
		return '';
	}
	$output = '';
	foreach ( $fields as $key => $value ) {
		$current = ( ! empty( $address ) && isset( $address[ $key ] ) ) ? $address[ $key ] : '';
		if ( isset( $_REQUEST['mc_address'] ) ) {
			$address = map_deep( $_REQUEST['mc_address'], 'sanitize_text_field' );
			$current = $address[ $key ];
		}
		$output .= '<p class="mc_field_' . sanitize_html_class( $key ) . '">
			<label for="mc_id_' . sanitize_html_class( $key ) . '">' . esc_html( $value['label'] ) . '</label> 
			<input type="text" autocomplete="' . esc_attr( $value['auto'] ) . '" value="' . $current . '" id="mc_id_' . sanitize_html_class( $key ) . '" name="mc_address[' . sanitize_html_class( $key ) . ']" />
		</p>';
	}
	$form = '
	<div class="mc-geolocate-form">
		<h2 class="screen-reader-text">' . esc_html__( 'Search Locations', 'my-calendar-pro' ) . '</h2>
		<form action="" method="POST">
			<input type="hidden" name="mc_gl_nonce" value="' . $nonce . '" />
			<div class="mc-gl-fields">
				' . $output . '
				<p class="mc-submit-button"><button class="button-primary mc-button">' . esc_html__( 'Find Locations', 'my-calendar-pro' ) . '</button></p>
			</div>
		</form>
	</div>';

	return $form;
}

/**
 * Get locations proximate to center point.
 *
 * @param float $latitude Center latitude.
 * @param float $longitude Center longitude.
 * @param array $address Array of address strings.
 * @param int   $radius Radius to search in kilometers.
 * @param int   $limit Number of results to display.
 *
 * @return array
 */
function mcs_geolocate( $latitude = false, $longitude = false, $address = array(), $radius = 500, $limit = 25 ) {
	global $wpdb;
	if ( isset( $_POST['mc_gl_nonce'] ) ) {
		$security = wp_verify_nonce( $_POST['mc_gl_nonce'], 'mc-geolocate-nonce' );
		if ( $security ) {
			$address = isset( $_REQUEST['mc_address'] ) ? map_deep( $_REQUEST['mc_address'], 'sanitize_text_field' ) : $address;
		}
	}
	if ( ! empty( $address ) ) {
		$coordinates = mc_get_location_coordinates( false, $address );
		$latitude    = $coordinates['latitude'];
		$longitude   = $coordinates['longitude'];
	}
	if ( ! $latitude || ! $longitude ) {
		$ip = false; // was mc_get_user_ip(). Disabled due to file size; GeoLite2 not included in package.
		if ( $ip ) {
			$reader    = new Reader( plugin_dir_path( __FILE__ ) . '/assets/GeoLite2-City.mmdb' );
			$response  = $reader->city( $ip );
			$latitude  = (float) $response->location->latitude;
			$longitude = (float) $response->location->longitude;
		} else {
			$latitude  = 0.000000;
			$longitude = 0.000000;
		}
	}
	$radius = (int) $radius;
	$limit  = (int) $limit;
	$output = '';
	// If units = miles, 3956 & 69; if km = 6371 & 111.0447
	// since MySQL 5.7.6; MariaDB 10.2.38; 10.3.29; 10.4.19; 10.5.10;
	// When ST Distance Sphere supported; .00062... = miles; .001 = km; default is meters.

	$locations = $wpdb->get_results( "SELECT *, ST_Distance_Sphere( point( $longitude, $latitude ), point(location_longitude, location_latitude)) * .000621371192 AS distance_in_miles FROM " . my_calendar_locations_table() . " HAVING distance_in_miles <= $radius ORDER BY distance_in_miles ASC LIMIT $limit" ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared
	if ( empty( $locations ) ) {
		$locations = $wpdb->get_results( "SELECT *, 3956 * 2 * ASIN(SQRT( POWER(SIN(($latitude - location_latitude)*pi()/180/2),2)+COS($latitude*pi()/180 )*COS(location_latitude*pi()/180)*POWER(SIN(($longitude-location_longitude)*pi()/180/2),2))) AS distance FROM " . my_calendar_locations_table() . " WHERE location_longitude BETWEEN ($longitude-$radius/cos(radians($latitude))*69) AND ($longitude+$radius/cos(radians($latitude))*69) AND location_latitude BETWEEN ($latitude-($radius/69)) AND ($latitude+($radius/69)) HAVING distance < $radius ORDER BY distance LIMIT $limit" ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared
	}
	if ( ! empty( $locations ) ) {
		$form   = mcs_geolocate_form( $address );
		$output = mc_generate_map( $locations, 'location', true, true );
	}

	return '<div class="mc-geolocate-wrapper">' . $form . $output . '</div>';
}
add_shortcode( 'geolocate', 'mcs_geolocation_map' );

/**
 * Generate a geolocation map based on latitude/longitude.
 *
 * @param array  $atts Shortcode attributes.
 * @param string $content Wrapped shortcode content.
 *
 * @return string
 */
function mcs_geolocation_map( $atts, $content ) {
	$args = shortcode_atts(
		array(
			'latitude'  => '',
			'longitude' => '',
			'radius'    => 100,
			'limit'     => 25,
		),
		$atts,
		'geolocate'
	);

	return mcs_geolocate( $args['latitude'], $args['longitude'], array(), $args['radius'], $args['limit'] );
}
