<?php
/**
 * Submission widgets
 *
 * @category Widgets
 * @package  My Calendar Pro
 * @author   Joe Dolson
 * @license  GPLv3
 * @link     https://www.joedolson.com/my-calendar-pro/
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * My Calendar Pro Submissions widget class.
 *
 * @category  Widgets
 * @package   My Calendar Pro
 * @author    Joe Dolson
 * @copyright 2012
 * @license   GPLv3
 * @version   1.0
 */
class Mc_Submissions_Widget extends WP_Widget {

	/**
	 * Constructor.
	 */
	public function __construct() {
		parent::__construct(
			false,
			$name = __( 'My Calendar: Submit an Event', 'my-calendar-pro' ),
			array( 'customize_selective_refresh' => true )
		);
	}

	/**
	 * Display widget.
	 *
	 * @param array $args Widget configuration.
	 * @param array $instance Current widget settings.
	 *
	 * @return void
	 */
	public function widget( $args, $instance ) {
		$before_widget = $args['before_widget'];
		$after_widget  = $args['after_widget'];
		$before_title  = str_replace( 'h1', 'h2', $args['before_title'] );
		$after_title   = str_replace( 'h1', 'h2', $args['after_title'] );
		$the_title     = apply_filters( 'widget_title', $instance['title'], $instance, $args );
		$form_id       = ( isset( $instance['form_id'] ) ) ? $instance['form_id'] : '';
		if ( ! $form_id ) {
			$fields          = ( isset( $instance['fields'] ) ) ? $instance['fields'] : array();
			$location_fields = ( isset( $instance['location_fields'] ) ) ? $instance['location_fields'] : array();
			$category        = ( isset( $instance['category'] ) ) ? esc_attr( $instance['category'] ) : 1; // set defaults.
			$location        = ( isset( $instance['location'] ) ) ? esc_attr( $instance['location'] ) : 1;
			$categories      = ( isset( $instance['categories'] ) ) ? esc_attr( $instance['categories'] ) : 'false'; // are they enabled.
			$locations       = ( isset( $instance['locations'] ) ) ? esc_attr( $instance['locations'] ) : 'false';
			$args            = array(
				'fields'          => $fields,
				'categories'      => $categories,
				'locations'       => $locations,
				'category'        => $category,
				'location'        => $location,
				'location_fields' => $location_fields,
				'form_id'         => 'widget_id_' . sanitize_title( $the_title ),
				'is_widget'       => true,
			);

			$args = mcs_convert_args( $args );
		} else {
			$args = array(
				'form_id'   => $form_id,
				'is_widget' => 1,
			);
		}
		$the_form = mcs_submit_form( $args );

		if ( mcs_user_can_submit_events() ) {
			$return  = $before_widget;
			$return .= ( $the_title ) ? $before_title . $the_title . $after_title : '';
			$return .= $the_form;
			$return .= $after_widget;
			// mc_kses_elements() added in My Calendar 3.3.0.
			if ( function_exists( 'mc_kses_elements' ) ) {
				echo wp_kses( $return, mc_kses_elements() );
			} else {
				echo wp_kses( $return, 'mycalendar' );
			}
		}
	}

	/**
	 * Generate widget form.
	 *
	 * @param array $instance Current data.
	 */
	public function form( $instance ) {
		mcs_check();
		$options      = mcs_get_default_settings();
		$defaults     = $options['widget_defaults'];
		$widget_title = ( ! empty( $instance['title'] ) ) ? esc_attr( $instance['title'] ) : $defaults['title'];
		$form_id      = ( isset( $instance['form_id'] ) ) ? $instance['form_id'] : '';
		if ( ! $form_id ) {
			$fields          = ( isset( $instance['fields'] ) ) ? $instance['fields'] : array();
			$location_fields = ( isset( $instance['location_fields'] ) ) ? $instance['location_fields'] : array();
			$category        = ( isset( $instance['category'] ) ) ? esc_attr( $instance['category'] ) : 1; // set defaults.
			$location        = ( isset( $instance['location'] ) ) ? esc_attr( $instance['location'] ) : 1;
			$categories      = ( isset( $instance['categories'] ) ) ? esc_attr( $instance['categories'] ) : 'false'; // are they enabled.
			$locations       = ( isset( $instance['locations'] ) ) ? esc_attr( $instance['locations'] ) : 'false';
			$args            = array(
				'fields'          => $fields,
				'categories'      => $categories,
				'locations'       => $locations,
				'category'        => $category,
				'location'        => $location,
				'location_fields' => $location_fields,
				'form_id'         => 'widget_id_' . sanitize_title( $widget_title ),
				'is_widget'       => true,
			);

			$args = mcs_convert_args( $args );
		}
		$link   = admin_url( 'admin.php?page=my-calendar-shortcodes#mc_builder' );
		$create = '<p><a href="' . esc_url( $link ) . '">' . __( 'Build an event submission form', 'my-calendar-pro' ) . '</a></p>';
		?>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"><?php esc_html_e( 'Widget Title', 'my-calendar-pro' ); ?></label><br />
			<input class="widefat" type="text" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" value="<?php echo esc_attr( $widget_title ); ?>"/>
		</p>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'form_id' ) ); ?>"><?php esc_html_e( 'Submission Form', 'my-calendar-pro' ); ?></label><br />
			<select id="<?php echo esc_attr( $this->get_field_id( 'form_id' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'form_id' ) ); ?>">
				<option value=""><?php esc_html_e( 'Select a form', 'my-calendar-pro' ); ?></option>
				<?php mcs_select_submission_forms( $form_id ); ?>
			</select>
		</p>
		<?php
		echo wp_kses_post( $create );
	}

	/**
	 * Update widget.
	 *
	 * @param array $new_settings New data.
	 * @param array $old_settings Old data.
	 *
	 * @return array
	 */
	public function update( $new_settings, $old_settings ) {
		$instance            = $old_settings;
		$instance['title']   = isset( $new_settings['title'] ) ? wp_kses_post( $new_settings['title'] ) : '';
		$instance['form_id'] = isset( $new_settings['form_id'] ) ? sanitize_key( $new_settings['form_id'] ) : '';

		return $instance;
	}
}
