<?php
/**
 * Settings for submissions
 *
 * @category Settings
 * @package  My Calendar Pro
 * @author   Joe Dolson
 * @license  GPLv3
 * @link     https://www.joedolson.com/my-calendar-pro/
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Update settings for payments & submissions.
 *
 * @param array $post POST data sanitized with wp_kses_post.
 *
 * @return string message
 */
function mcs_update_settings( $post ) {
	if ( isset( $post['mc-submit-settings'] ) ) {
		$nonce = $post['_wpnonce'];
		if ( ! wp_verify_nonce( $nonce, 'my-calendar-pro' ) ) {
			return;
		}
		$mcs_date_format        = $post['mcs_date_format'];
		$mcs_submit_id          = (int) $post['mcs_submit_id'];
		$mcs_edit_id            = (int) $post['mcs_edit_id'];
		$mcs_time_format        = $post['mcs_time_format'];
		$mcs_criteria           = ( isset( $post['mcs_criteria'] ) ) ? (int) $post['mcs_criteria'] : 2; // who can submit events.
		$mcs_check_conflicts    = ( isset( $post['mcs_check_conflicts'] ) ) ? 'true' : 'false'; // are conflicts a problem?
		$mcs_upload_images      = ( isset( $post['mcs_upload_images'] ) ) ? 'true' : 'false';
		$mcs_automatic_approval = ( isset( $post['mcs_automatic_approval'] ) ) ? 'true' : 'false'; // is approval required?
		$mcs_required_fields    = ( isset( $post['mcs_required_fields'] ) ) ? $post['mcs_required_fields'] : array();
		$mcs_ajax               = ( isset( $post['mcs_ajax'] ) ) ? 'true' : 'false';

		update_option( 'mcs_required_fields', $mcs_required_fields );
		update_option( 'mcs_check_conflicts', $mcs_check_conflicts );
		update_option( 'mcs_upload_images', $mcs_upload_images );
		update_option( 'mcs_automatic_approval', $mcs_automatic_approval );
		update_option( 'mcs_date_format', $mcs_date_format );
		update_option( 'mcs_submit_id', $mcs_submit_id );
		update_option( 'mcs_edit_id', $mcs_edit_id );
		update_option( 'mcs_time_format', $mcs_time_format );
		update_option( 'mcs_criteria', $mcs_criteria );
		update_option( 'mcs_ajax', $mcs_ajax );
		do_action( 'mcs_settings_update', $_POST );

		return __( 'My Calendar Event Submission Settings saved', 'my-calendar-pro' );
	}
	if ( isset( $post['mc-payment-settings'] ) ) {
		$nonce = $post['_wpnonce'];
		if ( ! wp_verify_nonce( $nonce, 'my-calendar-pro' ) ) {
			return;
		}
		$mcs_payments = ( isset( $post['mcs_payments'] ) ) ? 'true' : 'false'; // are payments required?
		if ( 'true' !== get_option( 'mcs_payments' ) && 'true' === $mcs_payments ) {
			mcs_update_database();
		}
		$mcs_gateway                      = ( isset( $post['mcs_gateway'] ) ) ? $post['mcs_gateway'] : 'paypal';
		$mcs_payment_response             = $post['mcs_payment_response']; // admin email after submission.
		$mcs_payment_confirmation         = $post['mcs_payment_confirmation']; // submitter email after submission.
		$mcs_payment_subject              = $post['mcs_payment_subject']; // subject line.
		$mcs_payment_message              = $post['mcs_payment_message'];
		$mcs_payment_confirmation_subject = $post['mcs_payment_confirmation_subject'];
		$mcs_use_sandbox                  = ( isset( $post['mcs_use_sandbox'] ) ) ? 'true' : 'false'; // Using sandbox?
		$mcs_submission_fee               = apply_filters( 'mcs_submission_fee', $post['mcs_submission_fee'] ); // posting cost for public.
		$mcs_members_discount             = (int) preg_replace( '/\D/', '', $post['mcs_members_discount'] );
		$mcs_paypal_email                 = is_email( $post['mcs_paypal_email'] ); // paypal email.
		$mcs_paypal_merchant_id           = $post['mcs_paypal_merchant_id']; // paypal merchant ID.
		$mcs_currency                     = $post['mcs_currency'];
		$mcs_discount                     = $post['mcs_discount']; // sets up default sales.

		update_option( 'mcs_payments', $mcs_payments );
		update_option( 'mcs_gateway', $mcs_gateway );
		update_option( 'mcs_discount', $mcs_discount );
		update_option( 'mcs_payment_response', $mcs_payment_response );
		update_option( 'mcs_payment_message', $mcs_payment_message );
		update_option( 'mcs_payment_confirmation', $mcs_payment_confirmation );
		update_option( 'mcs_payment_subject', $mcs_payment_subject );
		update_option( 'mcs_payment_confirmation_subject', $mcs_payment_confirmation_subject );
		update_option( 'mcs_use_sandbox', $mcs_use_sandbox );
		update_option( 'mcs_submission_fee', $mcs_submission_fee );
		update_option( 'mcs_members_discount', $mcs_members_discount );
		update_option( 'mcs_paypal_email', $mcs_paypal_email );
		update_option( 'mcs_paypal_merchant_id', $mcs_paypal_merchant_id );
		update_option( 'mcs_currency', $mcs_currency );
		do_action( 'mcs_payment_settings_update', $post );
		// Save Stripe config.
		$pk     = isset( $post['mcs_stripe_ppk'] ) ? $post['mcs_stripe_ppk'] : '';
		$tk     = isset( $post['mcs_stripe_tpk'] ) ? $post['mcs_stripe_tpk'] : '';
		$ps     = isset( $post['mcs_stripe_psk'] ) ? $post['mcs_stripe_psk'] : '';
		$ts     = isset( $post['mcs_stripe_tsk'] ) ? $post['mcs_stripe_tsk'] : '';
		$up     = isset( $post['mcs_stripe_update_webhooks'] ) ? $post['mcs_stripe_update_webhooks'] : false;
		$stripe = false;
		if ( $pk || $tk || $ps || $ts ) {
			$stripe = array(
				'prod_secret' => $ps,
				'test_secret' => $ts,
				'prod_public' => $pk,
				'test_public' => $tk,
			);
			if ( $up ) {
				$stripe['update_webhooks'] = $up;
			}
		}
		if ( $stripe ) {
			mcs_set_stripe_settings( $stripe );
			unset( $stripe['update_webhooks'] );
			update_option( 'mcs_stripe', $stripe );
		}

		__( 'My Calendar Payment Settings saved', 'my-calendar-pro' );
	}
	$custom_return = apply_filters( 'mcs_custom_settings_update', false, $_POST );
	if ( $custom_return ) {
		return $custom_return;
	}

	return false;
}

/**
 * When settings are saved, check for Stripe keys. If added or changed, create endpoint. Must be here; Stripe gateway not defined when disabled.
 *
 * @param array $new_options Array of Stripe keys.
 *
 * @return void
 */
function mcs_set_stripe_settings( $new_options ) {
	$old_options = get_option( 'mcs_stripe', array() );

	$test_secret_key  = trim( $new_options['test_secret'] );
	$test_osecret_key = isset( $old_options['test_secret'] ) ? trim( $old_options['test_secret'] ) : '';
	$live_secret_key  = trim( $new_options['prod_secret'] );
	$live_osecret_key = isset( $old_options['prod_secret'] ) ? trim( $old_options['prod_secret'] ) : '';

	$test_secret_key = ( $test_secret_key !== $test_osecret_key && '' !== $test_secret_key ) ? $test_secret_key : false;
	$live_secret_key = ( $live_secret_key !== $live_osecret_key && '' !== $live_secret_key ) ? $live_secret_key : false;

	$runsetup = false;
	if ( $new_options && isset( $new_options['update_webhooks'] ) ) {
		$runsetup = true;
		// If requesting runsetup, ensure there is an API key set.
		$test_secret_key = ( $test_secret_key !== $test_osecret_key && '' !== (string) $test_secret_key ) ? $test_secret_key : $test_osecret_key;
		$live_secret_key = ( $live_secret_key !== $live_osecret_key && '' !== (string) $live_secret_key ) ? $live_secret_key : $live_osecret_key;
	}

	if ( $test_secret_key || $runsetup ) {
		if ( $test_secret_key ) {
			\Stripe\Stripe::setApiKey( $test_secret_key );
			\Stripe\Stripe::setApiVersion( '2019-08-14' );

			$test_webhook = get_option( 'mcs_stripe_test_webhook', '' );
			if ( '' !== $test_webhook && $runsetup ) {
				$stripe = new \Stripe\StripeClient( $test_secret_key );
				// Check status.
				\Stripe\Stripe::setApiKey( $test_secret_key );
				\Stripe\Stripe::setApiVersion( '2019-08-14' );

				try {
					$stripe->webhookEndpoints->delete( $test_webhook, array() ); // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
				} catch ( Exception $e ) {
					if ( function_exists( 'mcs_log_error' ) ) {
						mcs_log_error( $e );
					}
				}
			}
			$endpoint = \Stripe\WebhookEndpoint::create(
				array(
					'url'            => add_query_arg( 'mcsipn', 'true', esc_url( home_url() ) ),
					'enabled_events' => array( '*' ),
					// Translators: Date webhook created.
					'description'    => sprintf( __( 'My Calendar Test Webhook %s', 'my-calendar-pro' ), date_i18n( 'Y-m-d H:i:s' ) ),
				)
			);
			update_option( 'mcs_stripe_test_webhook', $endpoint->id );
		}
	}

	if ( $live_secret_key || $runsetup ) {
		if ( $live_secret_key ) {
			\Stripe\Stripe::setApiKey( $live_secret_key );
			\Stripe\Stripe::setApiVersion( '2019-08-14' );

			$live_webhook = get_option( 'mcs_stripe_live_webhook', '' );
			if ( '' !== $live_webhook && $runsetup ) {
				$stripe = new \Stripe\StripeClient( $live_secret_key );
				try {
					$stripe->webhookEndpoints->delete( $live_webhook, array() ); // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
				} catch ( Exception $e ) {
					if ( function_exists( 'mcs_log_error' ) ) {
						mcs_log_error( $e );
					}
				}
			}
			$endpoint = \Stripe\WebhookEndpoint::create(
				array(
					'url'            => add_query_arg( 'mcsipn', 'true', esc_url( home_url() ) ),
					'enabled_events' => array( '*' ),
					// Translators: Date webhook created.
					'description'    => sprintf( __( 'My Calendar Live Webhook %s', 'my-calendar-pro' ), date_i18n( 'Y-m-d H:i:s' ) ),
				)
			);

			update_option( 'mcs_stripe_live_webhook', $endpoint->id );
		}
	}
}

/**
 * Setup settings for submissions & payments
 */
function mcs_settings() {
	mcs_check();
	$post     = map_deep( $_POST, 'wp_kses_post' );
	$response = mcs_update_settings( $post );
	if ( $response ) {
		wp_admin_notice(
			$response,
			array(
				'type' => 'success',
			)
		);
	}
	$options                          = mcs_get_default_settings();
	$defaults                         = $options['defaults'];
	$mcs_payments                     = ( get_option( 'mcs_payments' ) ) ? get_option( 'mcs_payments' ) : $defaults['mcs_payments']; // are payments required?
	$mcs_payment_subject              = ( get_option( 'mcs_payment_subject' ) ) ? get_option( 'mcs_payment_subject' ) : $defaults['mcs_payment_subject']; // subject line.
	$mcs_payment_response             = ( get_option( 'mcs_payment_response' ) ) ? get_option( 'mcs_payment_response' ) : $defaults['mcs_payment_response']; // admin email after submission.
	$mcs_payment_confirmation         = ( get_option( 'mcs_payment_confirmation' ) ) ? get_option( 'mcs_payment_confirmation' ) : $defaults['mcs_payment_confirmation']; // submitter email after submission.
	$mcs_payment_confirmation_subject = ( get_option( 'mcs_payment_confirmation_subject' ) ) ? get_option( 'mcs_payment_confirmation_subject' ) : $defaults['mcs_payment_confirmation_subject']; // subject line.
	$mcs_payment_message              = ( get_option( 'mcs_payment_message' ) ) ? get_option( 'mcs_payment_message' ) : $defaults['mcs_payment_message']; // subject line.
	$mcs_submission_fee               = ( get_option( 'mcs_submission_fee' ) ) ? get_option( 'mcs_submission_fee' ) : $defaults['mcs_submission_fee']; // posting cost for public.
	$mcs_members_discount             = get_option( 'mcs_members_discount' ); // discount for members (percentage).
	$mcs_criteria                     = ( '' !== get_option( 'mcs_criteria' ) ) ? get_option( 'mcs_criteria' ) : $defaults['mcs_criteria']; // who can submit events.
	$mcs_paypal_email                 = get_option( 'mcs_paypal_email' ); // paypal email.
	$mcs_use_sandbox                  = ( get_option( 'mcs_use_sandbox' ) ) ? get_option( 'mcs_use_sandbox' ) : $defaults['mcs_use_sandbox']; // use sandbox.
	$mcs_paypal_merchant_id           = get_option( 'mcs_paypal_merchant_id' ); // paypal merchant ID.
	$stripe                           = get_option( 'mcs_stripe' );
	$mcs_stripe_ppk                   = ( ! empty( $stripe ) ) ? $stripe['prod_public'] : '';
	$mcs_stripe_psk                   = ( ! empty( $stripe ) ) ? $stripe['prod_secret'] : '';
	$mcs_stripe_tpk                   = ( ! empty( $stripe ) ) ? $stripe['test_public'] : '';
	$mcs_stripe_tsk                   = ( ! empty( $stripe ) ) ? $stripe['test_secret'] : '';
	$mcs_submit_id                    = get_option( 'mcs_submit_id' );
	$mcs_edit_id                      = get_option( 'mcs_edit_id' );
	$mcs_date_format                  = get_option( 'mcs_date_format' );
	$mcs_time_format                  = get_option( 'mcs_time_format' );
	$mcs_currency                     = get_option( 'mcs_currency' );
	$mcs_discount                     = get_option(
		'mcs_discount',
		array(
			'begins' => '',
			'ends'   => '',
			'rate'   => '',
		)
	);
	$mcs_gateway                      = get_option( 'mcs_gateway' );
	$mcs_check_conflicts              = get_option( 'mcs_check_conflicts' );
	$mcs_upload_images                = get_option( 'mcs_upload_images' );
	$mcs_automatic_approval           = get_option( 'mcs_automatic_approval' );
	$mcs_required_fields              = get_option( 'mcs_required_fields', array() );
	$mcs_ajax                         = get_option( 'mcs_ajax', 'true' );
	?>
	<div class="wrap my-calendar-admin my-calendar-pro-settings" id="mc_settings">
		<h1><?php esc_html_e( 'My Calendar Pro', 'my-calendar-pro' ); ?></h1>
		<div class="mc-tabs vertical mcs-tabs mcs-settings settings postbox-container jcd-wide">
			<div class="tabs" role="tablist" data-default="mcs_tab">
				<button type="button" role="tab" aria-selected="false" id="tab_mcs" aria-controls="mcs_tab"><?php esc_html_e( 'Submissions', 'my-calendar-pro' ); ?></button>
				<button type="button" role="tab" aria-selected="false" id="tab_payments_mcs" aria-controls="mcs_payments_tab"><?php esc_html_e( 'Payments', 'my-calendar-pro' ); ?></button>
				<?php
				$tabs = apply_filters( 'mcs_settings_tabs', array() );
				foreach ( $tabs as $key => $value ) {
					?>
					<button type="button" role="tab" aria-selected="false" id="tab_<?php echo esc_attr( sanitize_title( $key ) ); ?>'_mcs" aria-controls="mcs_<?php echo esc_attr( sanitize_title( $key ) ); ?>_tab">
						<?php echo esc_html( $value ); ?>
					</button>
					<?php
				}
				?>
			</div>
			<div class="container">
			<div class="metabox-holder wptab" aria-labelledby="tab_mcs" role="tabpanel" id="mcs_tab">
				<form method="post" action="<?php echo esc_url( admin_url( 'admin.php?page=my-calendar-submissions' ) ); ?>">
				<div class="ui-sortable meta-box-sortables">
				<div class="postbox">
					<h2><?php esc_html_e( 'Event Submissions Settings', 'my-calendar-pro' ); ?></h2>
					<div class="inside">
						<?php
						if ( ! mcs_user_can_submit_events() ) {
							?>
							<div class="notice notice-warning inline"><p><?php echo wp_kses_post( __( 'Public Event Submissions are disabled. <a href="#mcs_public">Update Settings</a>', 'my-calendar-pro' ) ); ?></p></div>
							<?php
						} else {
							?>
						<div class="submissions-status">
							<ul>
							<?php
							$counts = get_option( 'mc_count_cache' );
							if ( ! empty( $counts['draft'] ) && 0 < absint( $counts['draft'] ) ) {
								?>
								<li><span class="review-pending"><span class="dashicons dashicons-edit" aria-hidden="true"></span><a href="<?php echo esc_url( admin_url( 'admin.php?page=my-calendar-manage&amp;limit=draft#my-calendar-admin-table' ) ); ?>"><?php esc_html_e( 'Review pending events', 'my-calendar-pro' ); ?></a></span></li>
								<?php
							} else {
								?>
								<li><span class="review-pending"><span class="dashicons dashicons-edit" aria-hidden="true"></span><?php esc_html_e( 'No pending events', 'my-calendar-pro' ); ?></span></li>
								<?php
							}
							?>
								<li><span class="dashicons dashicons-laptop" aria-hidden="true"></span>
								<?php
								if ( is_numeric( $mcs_submit_id ) && get_post( $mcs_submit_id ) ) {
									$link = '<a href="' . esc_url( get_the_permalink( $mcs_submit_id ) ) . '">';
									// Translators: Title of post to view.
									echo wp_kses_post( $link . __( 'Event submissions page', 'my-calendar-pro' ) . '</a>' );
								} else {
									echo ' <span class="mcs-no-page">';
									esc_html_e( 'No submission page set.', 'my-calendar-pro' );
									echo '</span>';
								}
								?>
								</li>
								<li><span class="dashicons dashicons-edit-page" aria-hidden="true"></span>
								<?php
								if ( is_numeric( $mcs_edit_id ) && get_post( $mcs_edit_id ) ) {
									$edit = '<a href="' . esc_url( get_the_permalink( $mcs_edit_id ) ) . '">';
									// Translators: Title of post to view.
									echo wp_kses_post( $edit . __( 'Event editing page', 'my-calendar-pro' ) . '</a>' );
								} else {
									echo ' <span class="mcs-no-page">';
									esc_html_e( 'No public event editing page set.', 'my-calendar-pro' );
									echo '</span>';
								}
								?>
								</li>
								<li><span class="customize-submissions"><span class="dashicons dashicons-admin-settings" aria-hidden="true"></span> <a href="<?php echo esc_url( admin_url( 'admin.php?page=my-calendar-shortcodes' ) ); ?>#mc_builder"><?php esc_html_e( 'Create new submissions form', 'my-calendar-pro' ); ?></a></span></li>
							</ul>
						</div>
							<?php
						}
						?>
						<div><input type="hidden" name="_wpnonce" value="<?php echo esc_attr( wp_create_nonce( 'my-calendar-pro' ) ); ?>" /></div>
						<p>
							<label for="mcs_submit_id"><?php esc_html_e( 'Event Submission Page ID', 'my-calendar-pro' ); ?></label><br>
							<input type="text" name="mcs_submit_id" id="mcs_submit_id" size="6" aria-describedby="mcs-submit-page" value="<?php echo esc_attr( trim( $mcs_submit_id ) ); ?>" /> <span id="mcs-submit-page">
							<?php
							if ( is_numeric( $mcs_submit_id ) && get_post( $mcs_submit_id ) ) {
								$link = '<a href="' . esc_url( get_edit_post_link( $mcs_submit_id ) ) . '">';
								// Translators: Title of post to edit.
								echo wp_kses_post( $link . sprintf( __( 'Edit "%s"', 'my-calendar-pro' ), esc_html( get_the_title( $mcs_submit_id ) ) ) . '</a>' );
							}
							if ( is_numeric( $mcs_submit_id ) && ! get_post( $mcs_submit_id ) ) {
								esc_html_e( 'Selected post ID does not exist.', 'my-calendar-pro' );
							}
							?>
							</span>
						</p>
						<p>
							<label for="mcs_edit_id"><?php esc_html_e( 'Event Editing Page ID', 'my-calendar-pro' ); ?></label><br />
							<input type="text" name="mcs_edit_id" id="mcs_edit_id" size="6" aria-describedby="mcs-edit-page" value="<?php echo esc_attr( trim( $mcs_edit_id ) ); ?>" /> <span id="mcs-edit-page">
							<?php
							if ( is_numeric( $mcs_edit_id ) && get_post( $mcs_edit_id ) ) {
								$link = '<a href="' . esc_url( get_edit_post_link( $mcs_edit_id ) ) . '">';
								// Translators: Title of post to edit.
								echo wp_kses_post( $link . sprintf( __( 'Edit "%s"', 'my-calendar-pro' ), esc_html( get_the_title( $mcs_edit_id ) ) ) . '</a>' );
							}
							if ( is_numeric( $mcs_edit_id ) && ! get_post( $mcs_edit_id ) ) {
								esc_html_e( 'Selected post ID does not exist.', 'my-calendar-pro' );
							}
							?>
							</span>
						</p>
						<p class='format'>
							<label for="mcs_date_format"><?php esc_html_e( 'Date format hint', 'my-calendar-pro' ); ?></label><br>
							<select name="mcs_date_format" id="mcs_date_format">
								<option value="m/d/Y" <?php selected( $mcs_date_format, 'm/d/y' ); ?>><?php echo esc_html( gmdate( 'm/d/Y' ) ); ?></option>
								<option value="d-m-Y" <?php selected( $mcs_date_format, 'd-m-Y' ); ?>><?php echo esc_html( gmdate( 'd-m-Y' ) ); ?></option>
								<option value="Y-m-d" <?php selected( $mcs_date_format, 'Y-m-d' ); ?>><?php echo esc_html( gmdate( 'Y-m-d' ) ); ?></option>
								<option value="j F Y" <?php selected( $mcs_date_format, 'j F Y' ); ?>><?php echo esc_html( date_i18n( 'j F Y' ) ); ?></option>
								<option value="M j, Y" <?php selected( $mcs_date_format, 'M j, Y' ); ?>><?php echo esc_html( gmdate( 'M j, Y' ) ); ?></option>
							</select>
						</p>
						<p class='format'>
							<label for="mcs_time_format"><?php esc_html_e( 'Time format hint', 'my-calendar-pro' ); ?></label><br>
							<select name="mcs_time_format" id="mcs_time_format">
								<option value="h:i a" <?php selected( $mcs_time_format, 'h:i a' ); ?>><?php echo esc_html( gmdate( 'h:i a' ) ); ?></option>
								<option value="H:i" <?php selected( $mcs_time_format, 'H:i' ); ?>><?php echo esc_html( gmdate( 'H:i' ) ); ?></option>
							</select>
						</p>
						<p>
							<input type="checkbox" id="mcs_ajax" name="mcs_ajax" <?php checked( $mcs_ajax, 'true' ); ?> /> <label for="mcs_ajax"><?php esc_html_e( 'Use AJAX form submission', 'my-calendar-pro' ); ?></label>
						</p>
						<p>
							<input type="checkbox" id="mcs_check_conflicts" name="mcs_check_conflicts" <?php checked( $mcs_check_conflicts, 'true' ); ?> /> <label for="mcs_check_conflicts"><?php esc_html_e( 'Prevent conflicting events. (if locations are used, checks only for conflicts at that location.)', 'my-calendar-pro' ); ?></label>
						</p>
						<p>
							<input type="checkbox" id="mcs_upload_images" name="mcs_upload_images" <?php checked( $mcs_upload_images, 'true' ); ?> /> <label for="mcs_upload_images"><?php esc_html_e( 'Allow public event submitters to upload their own images', 'my-calendar-pro' ); ?></label>
						</p>
						<p>
							<input type="checkbox" id="mcs_automatic_approval" name="mcs_automatic_approval" <?php checked( $mcs_automatic_approval, 'true' ); ?> /> <label for="mcs_automatic_approval"><?php esc_html_e( 'Automatically publish submitted events', 'my-calendar-pro' ); ?></label>
						</p>
					</div>
				</div>
			</div>
			<div class="ui-sortable meta-box-sortables">
				<div class="postbox">
					<h2><?php esc_html_e( 'Public Submission Required Fields', 'my-calendar-pro' ); ?></h2>
					<div class="inside">
						<fieldset>
							<legend class="screen-reader-text"><?php esc_html_e( 'Required fields', 'my-calendar-pro' ); ?></legend>
							<p><?php esc_html_e( 'Required fields will be added to any form that does not include them.', 'my-calendar-pro' ); ?></p>
							<?php
							$all_fields       = array(
								'event_title',
								'event_date',
								'event_end',
								'mcs_name',
								'mcs_email',
								'event_host',
								'event_recurring',
								'description',
								'short_description',
								'access',
								'event_link',
								'event_image',
								'categories',
								'registration',
								'custom_fields',
							);
							$mandatory_fields = array(
								'event_date',
								'event_time',
								'mcs_name',
								'mcs_email',
								'custom_fields',
								'access',
								'categories',
								'registration',
							);
							?>
							<ul class="checkboxes">
							<?php
							foreach ( $all_fields as $field ) {
								$search  = array( 'mcs_', 'event_', '_' );
								$replace = array( '', '', ' ' );
								$label   = ucfirst( str_replace( $search, $replace, $field ) );
								$checked = false;
								if ( in_array( $field, $mandatory_fields, true ) ) {
									continue;
								}
								if ( in_array( $field, $mcs_required_fields, true ) ) {
									$checked = true;
								}
								?>
								<li>
									<input type='checkbox' name='mcs_required_fields[]' value='<?php echo esc_attr( $field ); ?>' <?php checked( $checked, true ); ?> id='req_<?php echo esc_attr( $field ); ?>' />
									<label for='req_<?php echo esc_attr( $field ); ?>'><?php echo esc_html( $label ); ?></label>
								</li>
								<?php
							}
							?>
							</ul>
						</fieldset>
					</div>
				</div>
			</div>
			<div class="ui-sortable meta-box-sortables">
				<div class="postbox">
					<h2><?php esc_html_e( 'Event Submission Permissions', 'my-calendar-pro' ); ?></h2>
					<div class="inside">
						<fieldset>
							<legend class="screen-reader-text"><?php esc_html_e( 'Access to the public event submission form', 'my-calendar-pro' ); ?></legend>
							<ul>
							<li>
								<input type="radio" id="mcs_public" name="mcs_criteria" value="1" <?php checked( $mcs_criteria, '1' ); ?> /> <label for="mcs_public"><?php esc_html_e( 'General public.', 'my-calendar-pro' ); ?></label>
							</li>
							<li>
								<input type="radio" id="mcs_members_only" name="mcs_criteria" value="2" <?php checked( $mcs_criteria, '2' ); ?> /> <label for="mcs_members_only"><?php esc_html_e( 'Logged-in users', 'my-calendar-pro' ); ?></label>
							</li>
							<li>
								<input type="radio" id="mcs_member_status" name="mcs_criteria" value="3" <?php checked( $mcs_criteria, '3' ); ?> /> <label for="mcs_member_status"><?php esc_html_e( 'Users with the "mc_add_events" capability.', 'my-calendar-pro' ); ?></label>
							</li>
							<li>
								<input type="radio" id="mcs_public_disabled" name="mcs_criteria" value="0" <?php checked( $mcs_criteria, '0' ); ?> /> <label for="mcs_public_disabled"><?php esc_html_e( 'Front-end submissions disabled', 'my-calendar-pro' ); ?></label>
							</li>
							</ul>
						</fieldset>
					</div>
				</div>
			</div>
				<p><input type="submit" name="mc-submit-settings" class="button-primary" value="<?php esc_html_e( 'Save Submissions Settings', 'my-calendar-pro' ); ?>" /></p>
				</form>
			</div>
			<div class="metabox-holder wptab initial-hidden" aria-labelledby="tab_payments_mcs" role="tabpanel" id="mcs_payments_tab">
			<form method="post" action="<?php echo esc_url( admin_url( 'admin.php?page=my-calendar-submissions#mcs_payments_tab' ) ); ?>">
			<div class="ui-sortable meta-box-sortables">
				<div class="postbox">
					<h2><?php esc_html_e( 'Payment Settings', 'my-calendar-pro' ); ?></h2>
					<div class="inside">
					<?php
					// Requires SSL on server. Casual check.
					if ( false === stripos( home_url(), 'https' ) && 'true' === $mcs_payments ) {
						wp_admin_notice(
							__( 'Your website does not appear to be using an SSL certificate. Accepting payments requires your website to be configured over HTTPS.', 'my-calendar-pro' ),
							array(
								'type'               => 'error',
								'additional_classes' => array( 'mcs-ssl-warning' ),
							)
						);
					}
					?>
					<div><input type="hidden" name="_wpnonce" value="<?php echo esc_attr( wp_create_nonce( 'my-calendar-pro' ) ); ?>" /></div>
					<ul>
					<li>
						<input type="checkbox" id="mcs_payments" name="mcs_payments" <?php checked( $mcs_payments, 'true' ); ?> /> <label for="mcs_payments"><?php esc_html_e( 'Require payment to submit an event', 'my-calendar-pro' ); ?></label>
					</li>
					<?php
					if ( 'true' === $mcs_payments ) {
						?>
					<li>
						<input type="checkbox" id="mcs_use_sandbox" name="mcs_use_sandbox" <?php checked( $mcs_use_sandbox, 'true' ); ?> /> <label for="mcs_use_sandbox"><?php esc_html_e( 'Place gateway in Testing mode', 'my-calendar-pro' ); ?></label>
					</li>
					<li>
						<label for="mcs_payment_message"><?php esc_html_e( 'Payment Form Message (shows above payments form)', 'my-calendar-pro' ); ?></label> 
						<textarea class="widefat" type="text" name="mcs_payment_message" id="mcs_payment_message" rows="2" cols="60"><?php echo esc_textarea( wp_unslash( $mcs_payment_message ) ); ?></textarea>
						<em><?php echo wp_kses_post( __( 'Available template tags: <code>blogname</code>, <code>begins</code>, <code>ends</code>, <code>price</code>, <code>discount</code>, <code>currency</code>', 'my-calendar-pro' ) ); ?></em>
					</li>
					</ul>
				</div>
			</div>
			</div>
			<div class="ui-sortable meta-box-sortables">
				<div class="postbox">
					<h2><?php esc_html_e( 'New purchase messages', 'my-calendar-pro' ); ?></h2>
					<div class="inside">
					<fieldset>
						<legend><?php esc_html_e( 'Sent to administrators', 'my-calendar-pro' ); ?></legend>
						<ul>
							<li>
								<label for="mcs_payment_subject"><?php esc_html_e( 'Payment Notification Subject', 'my-calendar-pro' ); ?></label><br />
								<input type="text" name="mcs_payment_subject" id="mcs_payment_subject" class="widefat" value="<?php echo esc_attr( wp_unslash( $mcs_payment_subject ) ); ?>" />
							</li>
							<li>
								<label for="mcs_payment_response"><?php esc_html_e( 'Payment Notification message', 'my-calendar-pro' ); ?></label><br />
								<textarea class="widefat" name="mcs_payment_response" id="mcs_payment_response" rows="4" cols="60"><?php echo esc_textarea( wp_unslash( $mcs_payment_response ) ); ?></textarea>
								<em><?php echo wp_kses_post( __( 'Available template tags: <code>blogname</code>, <code>first_name</code>, <code>last_name</code>, <code>price</code>, <code>key</code>, <code>quantity</code>, <code>receipt</code>', 'my-calendar-pro' ) ); ?></em>
							</li>
						</ul>
					</fieldset>
					<fieldset>
						<legend><?php esc_html_e( 'Sent to purchaser', 'my-calendar-pro' ); ?></legend>
						<ul>
							<li>
								<label for="mcs_payment_confirmation_subject"><?php esc_html_e( 'Payment Confirmation Subject', 'my-calendar-pro' ); ?></label><br />
								<input type="text" name="mcs_payment_confirmation_subject" id="mcs_payment_confirmation_subject" class="widefat" value="<?php echo esc_attr( wp_unslash( $mcs_payment_confirmation_subject ) ); ?>" />
							</li>
							<li>
								<label for="mcs_payment_confirmation"><?php esc_html_e( 'Payment Submitter confirmation message', 'my-calendar-pro' ); ?></label><br />
								<textarea class="widefat" name="mcs_payment_confirmation" id="mcs_payment_confirmation" rows="4" cols="60"><?php echo esc_textarea( wp_unslash( $mcs_payment_confirmation ) ); ?></textarea>
								<em><?php echo wp_kses_post( __( 'Available template tags: <code>first_name</code>, <code>last_name</code>, <code>price</code>, <code>key</code>, <code>quantity</code>, <code>receipt</code>', 'my-calendar-pro' ) ); ?></em>
							</li>
						</ul>
					</fieldset>
				</div>
			</div>
		</div>
		<div class="ui-sortable meta-box-sortables">
			<div class="postbox">
				<h2><?php esc_html_e( 'Pricing', 'my-calendar-pro' ); ?></h2>
				<div class="inside">
					<ul>
						<li>
							<p>
							<label for="mcs_submission_fee"><?php esc_html_e( 'Base price:', 'my-calendar-pro' ); ?></label> 
							<input type="text" name="mcs_submission_fee" id="mcs_submission_fee" class="widefat" value="<?php echo esc_attr( $mcs_submission_fee ); ?>" />
							</p>
							<?php
								$pricing = apply_filters( 'mcs_submission_fee_settings', false );
								echo wp_kses( $pricing, mc_kses_elements() );
							?>
						</li>
						<li>
							<label for="mcs_currency"><?php esc_html_e( 'Currency:', 'my-calendar-pro' ); ?></label><br />
							<?php
							$mcs_currency_codes = array(
								'USD' => __( 'U.S. Dollars ($)', 'my-calendar-pro' ),
								'EUR' => __( 'Euros (€)', 'my-calendar-pro' ),
								'AUD' => __( 'Australian Dollars (A $)', 'my-calendar-pro' ),
								'CAD' => __( 'Canadian Dollars (C $)', 'my-calendar-pro' ),
								'GBP' => __( 'Pounds Sterling (£)', 'my-calendar-pro' ),
								'JPY' => __( 'Yen (¥)', 'my-calendar-pro' ),
								'NZD' => __( 'New Zealand Dollar ($)', 'my-calendar-pro' ),
								'CHF' => __( 'Swiss Franc', 'my-calendar-pro' ),
								'HKD' => __( 'Hong Kong Dollar ($)', 'my-calendar-pro' ),
								'SGD' => __( 'Singapore Dollar ($)', 'my-calendar-pro' ),
								'SEK' => __( 'Swedish Krona', 'my-calendar-pro' ),
								'DKK' => __( 'Danish Krone', 'my-calendar-pro' ),
								'PLN' => __( 'Polish Zloty', 'my-calendar-pro' ),
								'NOK' => __( 'Norwegian Krone', 'my-calendar-pro' ),
								'HUF' => __( 'Hungarian Forint', 'my-calendar-pro' ),
								'ILS' => __( 'Israeli Shekel', 'my-calendar-pro' ),
								'MXN' => __( 'Mexican Peso', 'my-calendar-pro' ),
								'BRL' => __( 'Brazilian Real (only for Brazilian users)', 'my-calendar-pro' ),
								'MYR' => __( 'Malaysian Ringgits (only for Malaysian users)', 'my-calendar-pro' ),
								'PHP' => __( 'Philippine Pesos', 'my-calendar-pro' ),
								'TWD' => __( 'Taiwan New Dollars', 'my-calendar-pro' ),
								'THB' => __( 'Thai Baht', 'my-calendar-pro' ),
								'TRY' => __( 'Turkish Lira (only for Turkish users)', 'my-calendar-pro' ),
							);
							echo "<select name='mcs_currency' id='mcs_currency'>";
							foreach ( $mcs_currency_codes as $code => $currency ) {
								$selected = ( $mcs_currency === $code ) ? true : false;
								?>
								<option value='<?php echo esc_attr( $code ); ?>' <?php selected( true, $selected ); ?>><?php echo esc_html( $currency ); ?></option>
								<?php
							}
							echo '</select>';
							?>
						</li>
						<li>
							<label for="mcs_members_discount"><?php esc_html_e( 'Member discount (%)', 'my-calendar-pro' ); ?></label><br /><input type="text" name="mcs_members_discount" id="mcs_members_discount" size="3" value="<?php echo esc_attr( $mcs_members_discount ); ?>" aria-describedby='mcs_member_discount_value' /> <span id="mcs_member_discount_value">
							<?php
							mcs_get_price( "Member's discounted cost:", 'my-calendar-pro' );
							echo ' ' . esc_html( mcs_get_price() );
							?>
							</span>
						</li>
						<?php echo wp_kses( apply_filters( 'mcs_custom_fields', '' ), mc_kses_elements() ); ?>
					</ul>
					</div>
				</div>
			</div>
			<div class="ui-sortable meta-box-sortables">
				<div class="postbox">
					<h2><?php esc_html_e( 'Payment Gateways', 'my-calendar-pro' ); ?></h2>
					<div class="inside">
					<ul class="checkboxes">
						<li>
							<input type="radio" id="mcs_gateway_paypal" name="mcs_gateway" value="paypal" <?php checked( $mcs_gateway, 'paypal' ); ?> /> <label for="mcs_gateway_paypal"><?php esc_html_e( 'Paypal', 'my-calendar-pro' ); ?></label>
						</li>
						<li>
							<input type="radio" id="mcs_gateway_stripe" name="mcs_gateway" value="stripe" <?php checked( $mcs_gateway, 'stripe' ); ?> /> <label for="mcs_gateway_stripe"><?php esc_html_e( 'Stripe', 'my-calendar-pro' ); ?></label>
						</li>
					</ul>
					<fieldset id="paypal" class="gateway">
						<legend><?php esc_html_e( 'PayPal Settings', 'my-calendar-pro' ); ?></legend>
						<ul>
							<li>
								<label for="mcs_paypal_email"><?php esc_html_e( 'Paypal email (primary)', 'my-calendar-pro' ); ?></label><br /><input type="text" name="mcs_paypal_email" id="mcs_paypal_email" class="widefat" value="<?php echo esc_attr( $mcs_paypal_email ); ?>" />
							</li>
							<li>
								<label for="mcs_paypal_merchant_id"><?php esc_html_e( 'Paypal merchant ID', 'my-calendar-pro' ); ?></label><br /><input type="text" name="mcs_paypal_merchant_id" id="mcs_paypal_merchant_id" class="widefat" value="<?php echo esc_attr( $mcs_paypal_merchant_id ); ?>" />
							</li>
						</ul>
					</fieldset>
					<fieldset id="stripe" class="gateway">
						<legend><?php esc_html_e( 'Stripe Settings', 'my-calendar-pro' ); ?></legend>
						<ul>
							<li>
								<label for="mcs_stripe_ppk"><?php esc_html_e( 'API Publishable Key (Production)', 'my-calendar-pro' ); ?></label><br /><input type="text" name="mcs_stripe_ppk" id="mcs_stripe_ppk" class="widefat" value="<?php echo esc_attr( $mcs_stripe_ppk ); ?>" />
							</li>
							<li>
								<label for="mcs_stripe_psk"><?php esc_html_e( 'API Secret Key (Production)', 'my-calendar-pro' ); ?></label><br /><input type="text" name="mcs_stripe_psk" id="mcs_stripe_psk" class="widefat" value="<?php echo esc_attr( $mcs_stripe_psk ); ?>" />
							</li>
							<li>
								<label for="mcs_stripe_tpk"><?php esc_html_e( 'API Publishable Key (Test)', 'my-calendar-pro' ); ?></label><br /><input type="text" name="mcs_stripe_tpk" id="mcs_stripe_tpk" class="widefat" value="<?php echo esc_attr( $mcs_stripe_tpk ); ?>" />
							</li>
							<li>
								<label for="mcs_stripe_tsk"><?php esc_html_e( 'API Secret Key (Test)', 'my-calendar-pro' ); ?></label><br /><input type="text" name="mcs_stripe_tsk" id="mcs_stripe_tsk" class="widefat" value="<?php echo esc_attr( $mcs_stripe_tsk ); ?>" />
							</li>
							<li>
								<input type="checkbox" name="mcs_stripe_update_webhooks" id="mcs_stripe_update_webhooks" value="true" /> <label for="mcs_stripe_update_webhooks"><?php esc_html_e( 'Update Stripe webhooks', 'my-calendar-pro' ); ?></label>
							</li>
						</ul>
						<?php
						if ( function_exists( 'mcs_stripe_setup_status' ) ) {
							echo wp_kses_post( mcs_stripe_setup_status() );
						}
						?>
					</fieldset>
					</div>
				</div>
			</div>
			<div class="ui-sortable meta-box-sortables">
				<div class="postbox">
					<h2><?php esc_html_e( 'Configure a Sale', 'my-calendar-pro' ); ?></h2>
					<div class="inside">
					<fieldset>
						<legend><?php esc_html_e( 'Sale settings', 'my-calendar-pro' ); ?></legend>
						<?php
						$sales = apply_filters( 'mcs_custom_sale', false );
						if ( ! $sales ) {
							?>
							<ul>
								<li>
									<label for="mcs_sale_begins"><?php esc_html_e( 'Date sale begins', 'my-calendar-pro' ); ?></label><br />
									<input type="text" name="mcs_discount[begins]" id="mcs_sale_begins" size="20" placeholder="<?php echo esc_attr( date_i18n( 'Y-m-d' ) ); ?>" aria-describedby="estimated_cost" value="<?php echo ( isset( $mcs_discount['begins'] ) ) ? esc_attr( $mcs_discount['begins'] ) : ''; ?>" /> <span id="estimated_cost"><?php esc_html_e( 'Discounted cost per event:', 'my-calendar-pro' ); ?> <?php echo esc_html( sprintf( '$%01.2f', mcs_get_price( true ) ) ); ?></span>
								</li>
								<li>
									<label for="mcs_sale_ends"><?php esc_html_e( 'Date sale ends', 'my-calendar-pro' ); ?></label><br />
									<input type="text" name="mcs_discount[ends]" id="mcs_sale_ends" size="20" value="<?php echo ( isset( $mcs_discount['ends'] ) ) ? esc_attr( $mcs_discount['ends'] ) : ''; ?>" />
								</li>
								<li>
									<label for="mcs_sale_rate"><?php esc_html_e( 'Percentage discount', 'my-calendar-pro' ); ?></label><br />
									<input type="text" name="mcs_discount[rate]" id="mcs_sale_rate" size="3" aria-describedby="discount_note" value="<?php echo ( isset( $mcs_discount['rate'] ) ) ? esc_attr( $mcs_discount['rate'] ) : ''; ?>" />
								</li>
							</ul>
							<?php
						} else {
							echo wp_kses( $sales, mc_kses_elements() );
						}
						?>
						<p id="discount_note"><?php echo wp_kses_post( __( '<strong>Note:</strong> if members have a discount, the additional sale rate will not be compounded with their normal rate.', 'my-calendar-pro' ) ); ?></p>
					</fieldset>
						<?php
					} else {
						?>
						<div class="mcs-about">
							<p>
								<?php esc_html_e( 'With payments enabled, visitors can purchase access to post events to your site. Enable payments to configure pricing and payment options.', 'my-calendar-pro' ); ?>
							</p>
						</div>
						<div>
						<input type="hidden" name="mcs_payment_response" value="<?php echo esc_attr( $mcs_payment_response ); ?>" />
						<input type="hidden" name="mcs_payment_confirmation" value="<?php echo esc_attr( $mcs_payment_confirmation ); ?>" />
						<input type="hidden" name="mcs_payment_subject" value="<?php echo esc_attr( $mcs_payment_subject ); ?>" />
						<input type="hidden" name="mcs_payment_message" value="<?php echo esc_attr( $mcs_payment_message ); ?>" />
						<input type="hidden" name="mcs_payment_confirmation_subject" value="<?php echo esc_attr( $mcs_payment_confirmation_subject ); ?>" />
						<input type="hidden" name="mcs_submission_fee" value="<?php echo esc_attr( $mcs_submission_fee ); ?>" />
						<?php
						$pricing = apply_filters( 'mcs_submission_fee_settings', false );
						echo wp_kses( $pricing, mc_kses_elements() );
						?>
						<input type="hidden" name="mcs_members_discount" value="<?php echo esc_attr( $mcs_members_discount ); ?>" />
						<input type="hidden" name="mcs_paypal_email" value="<?php echo esc_attr( $mcs_paypal_email ); ?>" />
						<input type="hidden" name="mcs_paypal_merchant_id" value="<?php echo esc_attr( $mcs_paypal_merchant_id ); ?>" />
						<input type="hidden" name="mcs_currency" value="<?php echo esc_attr( $mcs_currency ); ?>" />
						<input type="hidden" name="mcs_discount[begins]" value="<?php echo esc_attr( $mcs_discount['begins'] ); ?>" />
						<input type="hidden" name="mcs_discount[ends]" value="<?php echo esc_attr( $mcs_discount['ends'] ); ?>" />
						<input type="hidden" name="mcs_discount[rate]" value="<?php echo esc_attr( $mcs_discount['rate'] ); ?>" />
						</div>
						<?php
					}
					?>
					<p><input type="submit" name="mc-payment-settings" class="button-primary" value="<?php esc_html_e( 'Save Payment Settings', 'my-calendar-pro' ); ?>" /></p>
					</div>
				</div>
			</div>
			</form>
			</div>
			<?php
			$panels = apply_filters( 'mcs_settings_panels', array() );
			foreach ( $panels as $key => $value ) {
				$content    = ( is_array( $value ) && isset( $value['content'] ) ) ? $value['content'] : $value;
				$after_form = ( is_array( $value ) && isset( $value['after_form'] ) ) ? $value['after_form'] : '';
				$label      = ( is_array( $value ) && isset( $value['label'] ) ) ? $value['label'] : __( 'Save Settings', 'my-calendar-pro' );
				?>
				<div class="metabox-holder wptab initial-hidden" aria-labelledby="tab_<?php echo esc_attr( $key ); ?>_mcs" role="tabpanel" id="mcs_<?php echo esc_attr( $key ); ?>_tab">
					<form method="post" action="<?php echo esc_url( admin_url( "admin.php?page=my-calendar-submissions#mcs_$key" . '_tab' ) ); ?>" enctype="multipart/form-data">
					<?php wp_nonce_field( $key, '_wpnonce', true, true ); ?>
						<div class="ui-sortable meta-box-sortables">
							<div class="postbox">
				<?php
				$middle = str_replace( '{submit}', '<p><input type="submit" name="' . $key . '_settings" class="button-primary" value="' . $label . '" /></p>', $content );
				echo wp_kses( $middle, mc_kses_elements() );
				?>
							</div>
						</div>
					</form>
					<?php echo wp_kses( $after_form, mc_kses_elements() ); ?>
				</div>
				<?php
			}
			?>
			</div>
		</div>
		<?php
		$forms = get_option( 'mcs_forms' );
		// Translators: Help URL.
		$info = sprintf( __( '<a href="%s" class="mc-form-builder-link">Event submissions form builder</a>', 'my-calendar-pro' ), admin_url( 'admin.php?page=my-calendar-shortcodes#mc_builder' ) );
		// Translators: Documentation URL.
		$docs_text = sprintf( __( '<a href="%s">My Calendar Pro documentation</a>', 'my-calendar-pro' ), 'https://docs.joedolson.com/my-calendar-pro/' );
		$add       = array(
			'My Calendar Pro Help'   => '<p>' . $info . '</p><p>' . $docs_text . '</p>',
			'Saved Submission Forms' => '<ul>' . mcs_submission_forms( $forms ) . '</ul>',
			'Responsive Mode'        => '<p>' . __( 'The file <code>mc-responsive.css</code> in your theme directory will replace the My Calendar PRO responsive stylesheet.', 'my-calendar-pro' ) . '</p>',
		);
		$enabled   = get_option( 'mcs_responsive_mode' );
		if ( 'true' !== $enabled ) {
			unset( $add['Responsive Mode'] );
		}
		mc_show_sidebar( '', $add, true );
		?>
	</div>
	<?php
}
