<?php
/**
 * Create shortcode generator for submissions forms.
 *
 * @category Submissions
 * @package  My Calendar Pro
 * @author   Joe Dolson
 * @license  GPLv3
 * @link     https://www.joedolson.com/my-calendar-pro/
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Output fields for the My Calendar Pro form builder.
 *
 * @param array $id Form ID.
 *
 * @return string
 */
function mcs_form_builder_fields( $id = '' ) {
	$structure = array();
	$shortcode = '[submit_event]';
	$forms     = get_option( 'mcs_forms' );
	if ( isset( $_GET['delete'] ) ) {
		if ( wp_verify_nonce( $_GET['_mc_wpnonce'], 'my-calendar-generator' ) ) {
			$id = sanitize_key( $_GET['delete'] );
			if ( isset( $forms[ $id ] ) ) {
				unset( $forms[ $id ] );
				update_option( 'mcs_forms', $forms );
				// translators: Form ID.
				mc_show_notice( sprintf( __( 'Form %s deleted.', 'my-calendar-pro' ), '<code>' . $id . '</code>' ), true, false, 'success' );
			} else {
				// translators: Form ID.
				mc_show_error( sprintf( __( 'Form %s does not exist.', 'my-calendar-pro' ), '<code>' . $id . '</code>' ) );
			}
		}
	}
	if ( $id ) {
		$structure = isset( $forms[ $id ] ) ? $forms[ $id ] : $structure;
		$shortcode = '[submit_event form_id="' . esc_html( $id ) . '"]';
	}
	if ( isset( $_POST['shortcode'] ) && 'form' === $_POST['shortcode'] ) {
		if ( ! wp_verify_nonce( $_POST['_mc_wpnonce'], 'my-calendar-generator' ) ) {
			wp_die( 'My Calendar Pro: Security check failed', 'my-calendar-pro' );
		}
		$post      = map_deep( $_POST, 'sanitize_text_field' );
		$structure = array();
		// Check custom fields index against post data. If a percent encoded id was removed, restore it.
		// Allows fields that have percent-encoded keys to be enabled.
		$custom_fields = get_option( 'mcs_custom_fields', array() );
		$keys          = array_keys( $custom_fields );
		foreach ( $post['mc_fields'] as $key => $value ) {
			if ( ! $value ) {
				unset( $post['mc_fields'][ $key ] );
			}
		}
		if ( count( $post['mc_fields'] ) !== count( $_POST['mc_fields'] ) ) {
			$diff = array_diff( $_POST['mc_fields'], $post['mc_fields'] );
			foreach ( $diff as $val ) {
				if ( in_array( $val, $keys, true ) ) {
					$post['mc_fields'][] = $val;
				}
			}
		}
		foreach ( $post['mc_fields'] as $key ) {
			if ( in_array( $key, array( 'categories', 'locations' ), true ) ) {
				$post[ $key ] = ( 'categories' === $key ) ? '1' : $post[ $key ];
			}
			$structure['fields'][ $key ] = ( isset( $post['mc_labels'][ $key ] ) && ! empty( $post['mc_labels'][ $key ] ) ) ? $post['mc_labels'][ $key ] : 'true';
		}
		if ( isset( $post['categories'] ) ) {
			$structure['category']   = absint( $post['category'] );
			$structure['categories'] = '1';
			if ( 'hidden' === $post['category-entry'] ) {
				// If enabling, but type is still set to hidden, shift to default.
				$structure['category_field']['entry_type'] = '';
			} else {
				$structure['category_field']['entry_type'] = ( in_array( $post['category-entry'], array( 'multiple', 'hidden' ), true ) ) ? $post['category-entry'] : 'single';
			}
			$structure['category_field']['field_label'] = ( '' !== $post['category-label'] ) ? sanitize_text_field( $post['category-label'] ) : '';
		} else {
			$structure['categories']                   = '0';
			$structure['category_field']['entry_type'] = 'hidden';
		}
		if ( isset( $post['locations'] ) && in_array( 'locations', array_keys( $structure['fields'] ), true ) ) {
			$structure['locations'] = sanitize_text_field( $post['locations'] );
			if ( 'none' === $post['locations'] ) {
				// If was disabled, and is being disabled, change to a visible input type.
				$structure['locations'] = 'choose';
			}
			if ( isset( $post['location'] ) ) {
				$structure['location'] = absint( $post['location'] );
			}
			if ( isset( $post['location_fields'] ) ) {
				$location_fields = array();
				foreach ( $post['location_fields'] as $key => $field ) {
					if ( isset( $field['active'] ) ) {
						$field_label                            = ! empty( $field['label'] ) ? $field['label'] : 'true';
						$field_required                         = ! empty( $field['required'] ) ? $field['required'] : 'false';
						$location_fields[ $key ]['field_label'] = $field_label;
						$location_fields[ $key ]['required']    = $field_required;
					}
				}
				$structure['location_fields'] = $location_fields;
			}
		} else {
			$structure['locations'] = 'none';
		}
		$id = sanitize_key( sanitize_title( $post['mcs_form_name'] ) );
		if ( isset( $post['mcs_form_title'] ) ) {
			$structure['title'] = $post['mcs_form_title'];
		}
		$forms[ $id ] = $structure;
		update_option( 'mcs_forms', $forms );
		$shortcode = '[submit_event form_id="' . esc_html( $id ) . '"]';
	}
	$base           = 'submit_event';
	$type           = 'form';
	$include_fields = mcs_builder_fields( $structure );
	$info           = wp_kses_post(
		'<div class="shortcode-preview"><p><label for="mc_shortcode">Shortcode</label><textarea readonly class="large-text readonly mc-shortcode-container" id="mc_shortcode_' . $type . '">' . $shortcode . '</textarea></p>
		<div class="mc-copy-button"><button type="button" class="button-primary mc-copy-to-clipboard" data-clipboard-target="#mc_shortcode_' . $type . '">' . __( 'Copy to clipboard', 'my-calendar-pro' ) . '</button><span class="mc-notice-copied">' . __( 'Shortcode Copied', 'my-calendar-pro' ) . '</span></div>
		<p><button data-type="' . $base . '" type="button" class="button button-secondary reset-my-calendar">' . __( 'Reset Shortcode', 'my-calendar-pro' ) . '</button></p></div>'
	);

	$form_list = mcs_submission_forms( $forms, $id );
	if ( '' !== $form_list ) {
		$form_list = '<div class="mc-form-selector"><h3>' . __( 'Edit Form', 'my-calendar-pro' ) . '</h3><ul class="tabs">' . $form_list . '</ul></div>';
	}
	$nonce      = wp_create_nonce( 'my-calendar-generator' );
	$text       = ( $id ) ? __( 'Update Form', 'my-calendar-pro' ) : __( 'Save Form', 'my-calendar-pro' );
	$delete     = ( $id ) ? '<a class="button-secondary delete" href="' . esc_url( add_query_arg( '_mc_wpnonce', $nonce, admin_url( 'admin.php?page=my-calendar-shortcodes&delete=' . $id ) ) ) . '#mc_builder">' . __( 'Delete Form', 'my-calendar-pro' ) . '</a>' : '';
	$title      = ( $id ) ? mcs_form_title( $id ) : $id;
	$action_url = ( $id ) ? add_query_arg( 'form_id', $id, admin_url( 'admin.php?page=my-calendar-shortcodes' ) ) : admin_url( 'admin.php?page=my-calendar-shortcodes' );
	$output     = $info . '
	<div class="mc-form-builder">
		' . $form_list . '
		<form action="' . esc_url( $action_url ) . '#mc_builder" method="POST" id="my-calendar-generate">
			<div><input type="hidden" name="_mc_wpnonce" value="' . $nonce . '"/></div>
			<div id="mc-generator" class="generator">
				<input type="hidden" name="shortcode" value="form" />
				<div class="mc-flex mc-form-title">
					<p>
						<label for="mcs_form_title">' . __( 'Form Title', 'my-calendar-pro' ) . '</label>
						<input type="text" class="widefat" name="mcs_form_title" id="mcs_form_title" value="' . esc_attr( $title ) . '" />
					</p>
					<p>
						<label for="mcs_form_name">' . __( 'Form ID', 'my-calendar-pro' ) . ' ' . mcs_required_text() . '</label>
						<input type="text" class="widefat" name="mcs_form_name" id="mcs_form_name" value="' . esc_attr( $id ) . '" required />
					</p>
					<p>
						<input type="submit" class="button-secondary" name="generator" value="' . esc_attr( $text ) . '" /> ' . $delete . '
					</p>
				</div>
				' . $include_fields . '
			</div>
			<p>
			<input type="submit" class="button-primary" name="generator" value="' . esc_attr( $text ) . '" />
			</p>
		</form>
	</div>';

	return $output;
}

/**
 * Set the title of a form.
 *
 * @param string $id Form ID.
 * @param string $title Form title.
 */
function mcs_set_title( $id, $title ) {
	$forms = get_option( 'mcs_forms' );
	if ( is_array( $forms ) && isset( $forms[ $id ] ) ) {
		$form          = $forms[ $id ];
		$form['title'] = $title;
	}
}

/**
 * Get form title.
 *
 * @param string $id Form ID.
 *
 * @return string
 */
function mcs_form_title( $id ) {
	$forms = get_option( 'mcs_forms' );
	$title = $id;
	if ( is_array( $forms ) && isset( $forms[ $id ] ) ) {
		$form  = $forms[ $id ];
		$title = isset( $form['title'] ) ? $form['title'] : '';
	}
	return $title;
}

/**
 * List Submissions Forms.
 *
 * @param array  $forms Array of created forms.
 * @param string $id Form ID.
 *
 * @return string
 */
function mcs_submission_forms( $forms, $id = false ) {
	$base_url  = admin_url( 'admin.php?page=my-calendar-shortcodes#mc_builder' );
	$current   = ( '' === $id ) ? ' aria-current="true"' : '';
	$form_list = '<li><span class="dashicons dashicons-plus" aria-hidden="true"></span> <a ' . $current . ' class="new-form-link" href="' . esc_url( $base_url ) . '">' . __( 'Create New Form', 'my-calendar-pro' ) . '</a></li>';
	if ( is_array( $forms ) ) {
		foreach ( $forms as $key => $form ) {
			$current    = ( $key === $id ) ? ' aria-current="true"' : '';
			$title      = ( isset( $form['title'] ) && ! empty( $form['title'] ) ) ? $form['title'] : $key;
			$form_list .= '<li><a ' . $current . 'href="' . esc_url( add_query_arg( 'form_id', $key, $base_url ) ) . '">' . esc_html( $title ) . '</a></li>';
		}
	}

	return $form_list;
}

/**
 * Submissions Forms as select <option> elements.
 *
 * @param string $selected Form ID.
 */
function mcs_select_submission_forms( $selected = false ) {
	$forms = get_option( 'mcs_forms' );
	foreach ( $forms as $key => $form ) {
		?>
		<option value="<?php echo esc_attr( $key ); ?>"<?php selected( $selected, $key ); ?>><?php echo esc_html( $key ); ?></option>
		<?php
	}
}

/**
 * Add tab for form builder.
 *
 * @param string $output Existing tabs.
 *
 * @return string
 */
function mcs_form_builder_tab( $output ) {
	return $output . "	<button type='button' id='tab_mc_builder' role='tab' aria-selected='false' aria-controls='mc_builder'>" . esc_html( __( 'Submissions Forms', 'my-calendar-pro' ) ) . '</button>';
}

/**
 * Add form builder tab content.
 *
 * @param string $output Existing tab content.
 *
 * @return string
 */
function mcs_form_builder_tab_content( $output ) {
	$form_id = isset( $_GET['form_id'] ) ? sanitize_key( $_GET['form_id'] ) : '';

	return $output . "<div class='wptab mc_builder' id='mc_builder' aria-live='assertive' role='tabpanel' aria-labelledby='tab_mc_builder'>" . mcs_form_builder_fields( $form_id ) . '</div>';
}

/**
 * List of fields for form builder. Incomplete.
 *
 * @param array $structure Array of fields in this form.
 *
 * @return string
 */
function mcs_builder_fields( $structure = array() ) {
	$standard_fields = mcs_default_fields();
	$required        = array(
		'fields' => array(
			'submitter'   => '',
			'event_title' => '',
			'event_date'  => '',
			'event_time'  => '',
		),
	);
	$structure       = ( ! $structure || ! is_array( $structure ) ) ? $required : $structure;
	// Get array keys so individual custom fields show in generator.
	$custom_fields = mcs_get_fields();
	$groups        = mcs_get_builder_groups( $structure );
	$fields        = array_merge( $standard_fields, $groups, $custom_fields );
	if ( isset( $structure['categories'] ) ) {
		$fields[] = 'categories';
	}
	if ( isset( $structure['locations'] ) ) {
		$fields[] = 'locations';
	}
	$field_keys = array_merge( array_keys( $structure['fields'] ), array_keys( $fields ) );
	$controls   = array();
	foreach ( $field_keys as $field ) {
		$data = isset( $fields[ $field ] ) ? $fields[ $field ] : false;
		if ( is_array( $data ) ) {
			$label = ( isset( $data['field_label'] ) ) ? $data['field_label'] : false;
		} else {
			continue;
		}
		if ( '' === $field || ! $label ) {
			continue;
		}

		if ( in_array( $field, array_keys( $custom_fields ), true ) ) {
			$data['custom'] = 'true';
		}
		if ( in_array( $field, array_keys( $structure['fields'] ), true ) || in_array( $field, array_keys( $structure ), true ) || 'event_title' === $field || 'event_date' === $field || 'submitter' === $field ) {
			if ( ! in_array( $field, array( 'categories', 'locations' ), true ) && isset( $structure['fields'][ $field ] ) && '' !== $structure['fields'][ $field ] ) {
				$data['custom_label'] = $structure['fields'][ $field ];
			}
			$data['active'] = 'true';
			if ( 'categories' === $field ) {
				$entry_type = ( isset( $structure['category_field']['entry_type'] ) ) ? $structure['category_field']['entry_type'] : false;
				if ( 'hidden' === $entry_type ) {
					$data['active'] = false;
				}
			}
			if ( 'locations' === $field ) {
				if ( 'none' === $structure['locations'] ) {
					$data['active'] = false;
				}
			}
		} else {
			$data['active'] = 'false';
		}
		$controls[ $field ] = $data;
	}
	$output   = '<div class="mc-sortable-update" aria-live="assertive"></div><ul class="mc-sortable" id="mc-sortable-builder">';
	$i        = 1;
	$active   = '';
	$inactive = '';
	foreach ( $controls as $k => $v ) {
		$k            = trim( $k );
		$class        = ( 'true' === $v['active'] ) ? 'visible' : 'hidden';
		$disabled     = ( 'true' === $v['active'] ) ? '' : 'disabled';
		$custom_label = ( isset( $v['custom_label'] ) && 'true' !== $v['custom_label'] ) ? $v['custom_label'] : '';
		$custom_class = ( isset( $v['custom'] ) ) ? 'mc-custom-field' : '';
		if ( $v ) {
			$label = isset( $v['field_label'] ) ? $v['field_label'] : false;
			if ( ! $label ) {
				continue;
			}
			// Translators: control to move down.
			$down_label = sprintf( __( 'Move %s Down', 'my-calendar-pro' ), $label );
			// Translators: control to move up.
			$up_label = sprintf( __( 'Move %s Up', 'my-calendar-pro' ), $label );
			if ( 'visible' === $class ) {
				// Translators: control to hide.
				$hide_label = sprintf( __( 'Hide %s', 'my-calendar-pro' ), $label );
				$dashicon   = 'visibility';
				$pressed    = 'false';
			} else {
				// Translators: control to hide.
				$hide_label = sprintf( __( 'Show %s', 'my-calendar-pro' ), $label );
				$dashicon   = 'hidden';
				$pressed    = 'true';
			}
			$is_group = ( isset( $v['contains'] ) || isset( $v['section_contains'] ) ) ? true : false;
			if ( $is_group ) {
				if ( isset( $v['section_contains'] ) && is_array( $v['section_contains'] ) ) {
					$editor = '';
					foreach ( $v['section_contains'] as $field_key => $field_structure ) {
						$contains_label        = $field_structure['field_label'];
						$contains_custom_label = isset( $structure['fields'][ $k ][ $field_key ] ) ? $structure['fields'][ $k ][ $field_key ] : '';
						$editor               .= "<p><label for='mc-field-label-$field_key'>" . __( 'Field Label', 'my-calendar-pro' ) . ' (' . esc_html( stripslashes( $contains_label ) ) . ')</label><input type="text" class="widefat" value="' . $contains_custom_label . '" name="mc_labels[' . $k . '][' . $field_key . ']"' . " id='mc-field-label-$field_key' /></p>";
					}
				} else {
					$editor = $v['contains'];
				}
			} else {
				$editor = "<p><label for='mc-field-label-$k'>" . __( 'Field Label', 'my-calendar-pro' ) . ' (' . esc_html( stripslashes( $label ) ) . ')</label><input type="text" class="widefat" value="' . esc_html( stripslashes( $custom_label ) ) . '" name="mc_labels[' . $k . ']"' . " id='mc-field-label-$k' /></p>";
			}

			// Translators: control to edit.
			$edit_label = sprintf( __( 'Edit %s', 'my-calendar-pro' ), $label );
			$hide       = ( in_array( $k, array_keys( $required['fields'] ), true ) ) ? '' : "<button class='hide' aria-pressed='$pressed' type='button'><span class='screen-reader-text'>" . esc_html( stripslashes( $hide_label ) ) . "</span><i class='dashicons dashicons-$dashicon' aria-hidden='true'></i></button>";
			$edit       = ( in_array( $k, array_keys( $custom_fields ), true ) ) ? '' : "<button class='edit' aria-expanded='false' type='button'><span class='screen-reader-text'>" . esc_html( stripslashes( $edit_label ) ) . "</span><i class='dashicons dashicons-edit' aria-hidden='true'></i></button>";
			$buttons    = "<i class='dashicons dashicons-move' aria-hidden='true'></i> <button class='up' type='button'><i class='dashicons dashicons-arrow-up' aria-hidden='true'></i><span class='screen-reader-text'>" . esc_html( stripslashes( $up_label ) ) . "</span></button> <button class='down' type='button'><i class='dashicons dashicons-arrow-down' aria-hidden='true'></i><span class='screen-reader-text'>" . esc_html( stripslashes( $down_label ) ) . '</span></button> ' . $edit . $hide;
			$buttons    = "<div class='mc-buttons'>$buttons</div>";
			$field      = "<div class='mc-field-editor'>$editor</div>";
			if ( 'visible' === $class ) {
				$active .= "<li class='ui-state-default $custom_class mc-$k mc-$class'>$buttons <code>$k</code> " . esc_html( stripslashes( $label ) ) . "<input type='hidden' $disabled name='mc_fields[]' value='$k' />$field</li>";
			} else {
				$inactive .= "<li class='ui-state-default $custom_class mc-$k mc-$class'>$buttons <code>$k</code> " . esc_html( stripslashes( $label ) ) . "<input type='hidden' $disabled name='mc_fields[]' value='$k' />$field</li>";
			}
			++$i;
		}
	}
	$output .= $active . $inactive . '</ul>';

	return $output;
}

/**
 * Return groups of fields for form builder.
 *
 * @param array $structure Structure of currently selected form.
 *
 * @return array
 */
function mcs_get_builder_groups( $structure ) {
	$location_fields = mcs_default_location_fields();
	$output          = '';
	foreach ( $location_fields as $field => $data ) {
		$field_title = $data['field_label'];
		$value       = '';
		$enabled     = ' checked="checked"';
		$required    = '';
		if ( isset( $structure['location_fields'][ $field ] ) ) {
			$value    = isset( $structure['location_fields'][ $field ]['field_label'] ) ? $structure['location_fields'][ $field ]['field_label'] : 'true';
			$value    = ( 'true' === $value ) ? '' : $value;
			$required = isset( $structure['location_fields'][ $field ]['required'] ) ? $structure['location_fields'][ $field ]['required'] : 'false';
		}
		if ( isset( $structure['location_fields'] ) && ! isset( $structure['location_fields'][ $field ] ) ) {
			$enabled = '';
		}
		if ( 'gps' === $field ) {
			$display = '<em>' . __( 'Custom label not available for compound fields.', 'my-calendar-pro' ) . '</em>';
		} else {
			// Translators: Standard name of field.
			$display = "<p><label for='label_$field'>" . sprintf( __( 'Custom label for "%s"', 'my-calendar-pro' ), $field_title ) . "</label><input class='widefat' type='text' name='location_fields[$field][label]' id='label_$field' value='" . esc_attr( $value ) . "' placeholder='" . esc_attr( $field_title ) . "' /></p>";
			// Translators: Standard name of field.
			$display .= "<p class='checkbox'><input type='checkbox' value='on' " . checked( $required, 'on', false ) . " name='location_fields[$field][required]' id='label_required_$field'> <label for='label_required_$field'>" . sprintf( __( '%s required', 'my-calendar-pro' ), $field_title ) . '</label></p>';
		}
		$output .= "
		<div class='mc-row'>
			<div class='mc-enable'>
			<input type='checkbox' $enabled value='on' name='location_fields[$field][active]' id='mc-builder-$field' /> 
			<label for='mc-builder-$field'>$field_title</label></div>
			<div class='mc-label'>$display</div>
		</div>";
	}
	$category_selected = ( isset( $structure['category'] ) ) ? $structure['category'] : false;
	$location_entry    = ( isset( $structure['locations'] ) ) ? $structure['locations'] : 'none';
	$location_selected = ( isset( $structure['location'] ) ) ? absint( $structure['location'] ) : false;
	$category_label    = ( isset( $structure['category_field']['field_label'] ) ) ? $structure['category_field']['field_label'] : '';
	$category_entry    = ( isset( $structure['category_field']['entry_type'] ) ) ? $structure['category_field']['entry_type'] : '';
	$groups            = array(
		'categories' => array(
			'field_label' => __( 'Category selector', 'my-calendar-pro' ),
			'contains'    => "
				<p>
					<label for='mc-builder-category'>" . __( 'Select default category', 'my-calendar-pro' ) . "</label>
					<select name='category' id='mc-builder-category'>" . mc_category_select( $category_selected ) . '</select>
				</p>
			 	<p>
					<label for="mc-builder-category-mode">' . __( 'Category entry type', 'my-calendar-pro' ) . '</label>
			 		<select name="category-entry" id="mc-builder-category-entry">
						<option value="">' . __( 'Default', 'my-calendar-pro' ) . '</option>
						<option value="hidden"' . selected( $category_entry, 'hidden', false ) . '>' . __( 'Hidden', 'my-calendar-pro' ) . '</option>
						<option value="multiple"' . selected( $category_entry, 'multiple', false ) . '>' . __( 'Multiple', 'my-calendar-pro' ) . '</option>
						<option value="single"' . selected( $category_entry, 'single', false ) . '>' . __( 'Single', 'my-calendar-pro' ) . '</option>
					</select>
				</p>
				<p>
					<label for="mc-builder-category-label">' . __( 'Category Field Label', 'my-calendar-pro' ) . '</label>
					<input type="text" id="mc-builder-category-label" name="category-label" value="' . esc_attr( $category_label ) . '" />
				</p>',
		),
		'locations'  => array(
			'field_label' => __( 'Location selector', 'my-calendar-pro' ),
			'contains'    => "<fieldset class='mc-flex'><legend>" . __( 'Location Inputs', 'my-calendar-pro' ) . "</legend>
				<p><label for='mc-builder-locations'>" . __( 'Enable Location Options', 'my-calendar-pro' ) . "</label>
				<select name='locations' id='mc-builder-locations'>
					<option value='none'" . selected( $location_entry, 'none', false ) . '>' . __( 'Locations disabled', 'my-calendar-pro' ) . "</option>
					<option value='choose'" . selected( $location_entry, 'choose', false ) . '>' . __( 'User can choose an existing location', 'my-calendar-pro' ) . "</option>
					<option value='enter'" . selected( $location_entry, 'enter', false ) . '>' . __( 'User can enter new locations', 'my-calendar-pro' ) . "</option>
					<option value='either'" . selected( $location_entry, 'either', false ) . '>' . __( 'User can either enter new locations or select an existing location', 'my-calendar-pro' ) . '</option>
				</select>
				</p>
				<p><label for="mc-builder-location">' . __( 'Select default location', 'my-calendar-pro' ) . "</label>
				<select name='location' id='mc-builder-location'>" . mc_location_select( $location_selected ) . '</select></p>
				</fieldset>
				<fieldset>
					<legend>' . __( 'Enable Location Fields', 'my-calendar-pro' ) . '</legend>
					<div class="mc-generator">' . $output . '</div>
				</fieldset>',
		),
	);

	return $groups;
}
