<?php
/**
 * Private Content in Events
 *
 * @category Events
 * @package  My Calendar Pro
 * @author   Joe Dolson
 * @license  GPLv3
 * @link     https://www.joedolson.com/my-calendar-pro/
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

add_filter( 'mc_inner_content', 'mcp_event_content', 10, 4 );
/**
 * Hide content details if user does not have sufficient privileges.
 *
 * @param string $details Event description block.
 * @param object $event Event object.
 * @param string $type Display type.
 * @param string $time Current time format.
 *
 * @return string
 */
function mcp_event_content( $details, $event, $type, $time ) {
	$post         = $event->event_post;
	$hide_details = get_post_meta( $post, '_hide_event_details', true );
	$user_can_see = ( current_user_can( 'mc_manage_events' ) || current_user_can( 'mc_view_hidden_events' ) ) ? true : false;
	$user_can_see = apply_filters( 'mc_user_can_see_content', $user_can_see, $event );
	if ( ( 'true' === $hide_details && $user_can_see ) || 'true' !== $hide_details ) {
		$details = $details;
	} else {
		$details = '<div class="details ' . $type . '-details mc-content-private">' . wpautop( apply_filters( 'my_calendar_hidden_content', __( 'Log-in to see details about this event.', 'my-calendar-pro' ), $event, $type, $time ) ) . '</div>';
	}

	return $details;
}


/**
 * Insert private content setting on misc settings tab.
 *
 * @param array $fields Miscellaneous settings fields.
 *
 * @return array
 */
function mcs_private_content_setting( $fields ) {
	$fields['private_content'] = array(
		'field_label'       => __( 'Enable private event description setting', 'my-calendar-pro' ),
		'field_type'        => 'checkbox',
		'field_options'     => 'true',
		'id'                => 'private_content',
		'sanitize_callback' => 'sanitize_text_field',
		'field_default'     => ( 'true' === get_option( 'mcs_private_content', 'false' ) ) ? 'true' : '',
	);

	return $fields;
}
add_filter( 'mcs_miscellaneous_settings', 'mcs_private_content_setting', 10, 1 );

/**
 * Insert custom permission for My Calendar private content.
 *
 * @param array $caps Existing capabilities.
 *
 * @return array
 */
function mcp_event_content_privileges( $caps ) {
	$caps['mc_view_hidden_events'] = __( 'View Hidden Event Details', 'my-calendar-pro' );

	return $caps;
}
add_filter( 'mc_capabilities', 'mcp_event_content_privileges' );

add_action( 'mc_update_event_post', 'mcp_hide_event_data_save', 10, 4 );
/**
 * Update event post to indicate whether data should be hidden.
 *
 * @param integer $post_id Post ID.
 * @param array   $post POST data.
 * @param array   $data Event data.
 * @param integer $event_id Event ID.
 */
function mcp_hide_event_data_save( $post_id, $post, $data, $event_id ) {
	$content = isset( $_POST['mcp_hide_event_details'] ) ? 'true' : '';
	update_post_meta( $post_id, '_hide_event_details', $content );
}

add_filter( 'mc_show_block', 'mcp_hide_event', 10, 3 );
/**
 * Show checkbox about hiding events.
 *
 * @param string $form Block form.
 * @param object $event Event data.
 * @param string $field current field.
 *
 * @return string.
 */
function mcp_hide_event( $form, $event, $field ) {
	// If this setting is disabled or not set, return original form.
	if ( 'false' === get_option( 'mcs_private_content' ) ) {
		return $form;
	}
	if ( 'event_category' === $field ) {
		if ( ! ( is_object( $event ) && property_exists( $event, 'event_post' ) ) ) {
			$checked = '';
		} else {
			$checked = ( 'true' === get_post_meta( $event->event_post, '_hide_event_details', true ) ) ? 'checked="checked"' : '';
		}
		$form = "
		<div class='mcp-private-event-details'>
			<p class='checkbox'>
				<input type='checkbox' name='mcp_hide_event_details' id='mcp_hide_event_details' value='true' $checked /> <label for='mcp_hide_event_details'>" . __( 'Keep Event Details Private', 'my-calendar-pro' ) . '</label>
			</p>
		</div>' . $form;
	}

	return $form;
}
