<?php
/**
 * Multilingual Events
 *
 * @category Events
 * @package  My Calendar Pro
 * @author   Joe Dolson
 * @license  GPLv3
 * @link     https://www.joedolson.com/my-calendar-pro/
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

add_action( 'my_calendar_drawing_event', 'mcp_switch_locale' );
add_action( 'mc_create_tags', 'mcp_switch_locale', 10, 1 );
add_action( 'my_calendar_event_drawn', 'mcp_restore_locale', 10, 1 );
add_action( 'mc_tags_created', 'mcp_restore_locale' );
add_filter( 'mc_event_details_output', 'mcp_event_language', 10, 2 );
add_filter( 'mc_template', 'mcp_event_template_language', 10, 4 );
add_filter( 'mc_event_object', 'mcp_event_object' );

/**
 * Set event language parameters on event object.
 *
 * @param object $event Event object.
 *
 * @return object
 */
function mcp_event_object( $event ) {
	$locale    = get_locale();
	$eventlang = get_post_meta( $event->event_post, '_event_language', true );

	$event->site_locale  = $locale;
	$event->event_locale = $eventlang;

	return $event;
}

/**
 * If event is in an alternate locale, switch to locale.
 *
 * @param object $event Event object.
 */
function mcp_switch_locale( $event ) {
	if ( property_exists( $event, 'site_locale' ) ) {
		$locale    = $event->site_locale;
		$eventlang = $event->event_locale;
		if ( $eventlang && ( $locale !== $eventlang ) ) {
			switch_to_locale( $eventlang );
		}
	}
}

/**
 * After rendering event, restore previous locale.
 *
 * @param object $event Event object.
 */
function mcp_restore_locale( $event ) {
	if ( property_exists( $event, 'site_locale' ) ) {
		$locale    = $event->site_locale;
		$eventlang = $event->event_locale;
		if ( $eventlang && ( $locale !== $eventlang ) ) {
			switch_to_locale( $locale );
		}
	}
}

/**
 * Wrap event output in a language container.
 *
 * @param string $details Event HTML output.
 * @param object $event Event object.
 *
 * @return string
 */
function mcp_event_language( $details, $event ) {
	$post_ID  = $event->event_post;
	$language = get_post_meta( $post_ID, '_event_language', true );
	if ( $language ) {
		$lang    = explode( '_', $language )[0];
		$details = ( $details ) ? '<div class="mc-language" lang="' . esc_attr( $lang ) . '">' . $details . '</div>' : '';
	}

	return $details;
}

/**
 * Wrap event output in a language container.
 *
 * @param string $template Event HTML output.
 * @param array  $details Event data array.
 * @param string $type Type of template.
 * @param object $event Event object.
 *
 * @return string
 */
function mcp_event_template_language( $template, $details, $type, $event ) {
	if ( is_object( $event ) ) {
		if ( property_exists( $event, 'site_locale' ) ) {
			$locale    = $event->site_locale;
			$eventlang = $event->event_locale;
			if ( $eventlang && ( $locale !== $eventlang ) ) {
				switch_to_locale( $eventlang );
			}
		}
		if ( $eventlang ) {
			$lang     = explode( '_', $eventlang )[0];
			$template = '<div class="mc-language" lang="' . esc_attr( $lang ) . '">' . $template . '</div>';
		}
		if ( property_exists( $event, 'site_locale' ) ) {
			if ( $eventlang && ( $locale !== $eventlang ) ) {
				switch_to_locale( $locale );
			}
		}
	}

	return $template;
}

add_action( 'mc_update_event_post', 'mcp_event_language_save', 10, 2 );
/**
 * Update event post to indicate the event language
 *
 * @param integer $post_id Post ID.
 * @param array   $post POST data.
 */
function mcp_event_language_save( $post_id, $post ) {
	$content = isset( $post['locale'] ) ? sanitize_text_field( $post['locale'] ) : '';
	if ( $content ) {
		update_post_meta( $post_id, '_event_language', $content );
	}
}

add_filter( 'mc_insert_custom_fields', 'mcp_i18n_event', 10, 3 );
/**
 * Show select field to set language of event.
 *
 * @param string $form Block form.
 * @param object $has_data Is data assigned..
 * @param string $data Event data.
 *
 * @return string|void
 */
function mcp_i18n_event( $form, $has_data, $data ) {
	$languages = get_available_languages();
	if ( 'true' !== get_option( 'mcs_multilingual_events', 'false' ) ) {
		return '';
	}
	if ( version_compare( get_option( 'mc_version' ), '3.3.14', '>' ) && count( $languages ) > 0 ) {
		if ( ! ( is_object( $data ) && property_exists( $data, 'event_post' ) ) ) {
			// Default to the site's defined language.
			$selected = get_bloginfo( 'language' );
		} else {
			$selected = get_post_meta( $data->event_post, '_event_language', true );
		}
		$args         = array(
			'echo'                        => 0,
			'languages'                   => $languages,
			'selected'                    => $selected,
			'show_available_translations' => 0,
		);
		$translations = str_replace( 'id="locale"', 'aria-describedby="langnote" id="locale"', wp_dropdown_languages( $args ) );

		$form = "
		<div class='mcp-event-language'>
			<p class='select'>
				<label for='locale'>" . __( 'Event Listing Language', 'my-calendar-pro' ) . "</label><br />
				$translations
				<span id='langnote' class='mc-help-note'><i class='dashicons dashicons-editor-help' aria-hidden='true'></i>" . __( 'What language is this listing written in?', 'my-calendar-pro' ) . '</span>
			</p>
		</div>';

		echo wp_kses( $form, mc_kses_elements() );
	}
}

/**
 * Insert multilingual setting on misc settings tab.
 *
 * @param array $fields Miscellaneous settings fields.
 *
 * @return array
 */
function mcs_multilingual_setting( $fields ) {
	$fields['multilingual'] = array(
		'field_label'       => __( 'Enable language selector on events', 'my-calendar-pro' ),
		'field_type'        => 'checkbox',
		'field_options'     => 'true',
		'id'                => 'multilingual_events',
		'sanitize_callback' => 'sanitize_text_field',
		'field_default'     => ( 'true' === get_option( 'mcs_multilingual_events', 'false' ) ) ? 'true' : '',
	);

	return $fields;
}
add_filter( 'mcs_miscellaneous_settings', 'mcs_multilingual_setting', 10, 1 );
