<?php
/**
 * My Calendar REST API Route
 *
 * @category REST API
 * @package  My Calendar Pro
 * @author   Joe Dolson
 * @license  GPLv3
 * @link     https://www.joedolson.com/my-calendar-pro/
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * My Calendar REST API Route.
 *
 * @category  REST API
 * @package   My Calendar Pro
 * @author    Joe Dolson
 * @copyright 2012
 * @license   GPLv3
 * @version   1.0
 */
class My_Calendar_API_Route extends WP_REST_Controller {

	/**
	 * Register the routes for the objects of the controller.
	 */
	public function register_routes() {
		$version   = '1';
		$namespace = 'mc_api/v' . $version;
		$base      = 'events';

		register_rest_route(
			$namespace,
			'/' . $base,
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'create_item' ),
					'permission_callback' => array( $this, 'create_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema(),
				),
			)
		);

		register_rest_route(
			$namespace,
			'/' . $base . '/(?P<id>[\d]+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'context' => array(
							'default' => 'view',
						),
					),
				),
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'update_item' ),
					'permission_callback' => array( $this, 'update_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema(),
				),
				array(
					'methods'             => WP_REST_Server::DELETABLE,
					'callback'            => array( $this, 'delete_item' ),
					'permission_callback' => array( $this, 'delete_item_permissions_check' ),
					'args'                => array(
						'force' => array(
							'default' => false,
						),
					),
				),
			)
		);

		register_rest_route(
			$namespace,
			'/' . $base . '/schema',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_public_item_schema' ),
				'permission_callback' => array( $this, 'get_item_permissions_check' ),
			)
		);
	}

	/**
	 * Get a collection of items
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return WP_Error|WP_REST_Response
	 */
	public function get_items( $request ) {
		$params = $request->get_params();

		$from             = isset( $params['from'] ) ? $params['from'] : gmdate( 'Y-m-d', current_time( 'timestamp' ) - 2 * WEEK_IN_SECONDS ); // phpcs:ignore WordPress.DateTime.CurrentTimeTimestamp.Requested
		$to               = isset( $params['to'] ) ? $params['from'] : gmdate( 'Y-m-d', current_time( 'timestamp' ) + 2 * WEEK_IN_SECONDS ); // phpcs:ignore WordPress.DateTime.CurrentTimeTimestamp.Requested
		$params['source'] = 'api';
		$params['from']   = $from;
		$params['to']     = $to;

		$events = my_calendar_events( $params );
		$data   = array();
		foreach ( $events as $item ) {
			$itemdata = $this->prepare_item_for_response( $item, $request );
			$data[]   = $this->prepare_response_for_collection( $itemdata );
		}

		return new WP_REST_Response( $data, 200 );
	}

	/**
	 * Get one item from the collection
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return WP_Error|WP_REST_Response
	 */
	public function get_item( $request ) {
		// get parameters from request.
		$params = $request->get_params();

		$item = mc_get_event( $params['id'] ); // do a query, call another class, etc.
		$data = $this->prepare_item_for_response( $item, $request );

		// return a response or error based on some conditional.
		if ( $data ) {

			return new WP_REST_Response( $data, 200 );
		} else {

			return new WP_Error(
				'cant-retrieve',
				__( 'The requested event does not exist or is not available.', 'my-calendar-pro' ),
				array( 'status' => 404 )
			);
		}
	}

	/**
	 * Create one item from the collection
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return WP_Error|WP_REST_Request
	 */
	public function create_item( $request ) {
		$item        = $this->prepare_item_for_database( $request );
		$params      = $request->get_params();
		$original_id = $params['event_id'];

		if ( function_exists( 'mcs_api_create_item' ) && false !== $item[0] ) {
			$data = mcs_api_create_item( $request, $item, $original_id );
			if ( is_array( $data ) ) {
				return new WP_REST_Response( $data, 200 );
			}
		}
		$error = ( '' === $item[3] ) ? __( 'Unable to create event. Invalid data submitted.', 'my-calendar-pro' ) : $item[3];

		return new WP_Error(
			'cant-create',
			$error,
			array( 'status' => 500 )
		);
	}

	/**
	 * Update one item from the collection
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return WP_Error|WP_REST_Request
	 */
	public function update_item( $request ) {
		$item   = $this->prepare_item_for_database( $request );
		$params = $request->get_params();
		// check whether this indicated item exists on this server.
		// if exists, return correct ID to update, else false.
		$update_id = mcs_api_check_requested_id( $params );

		if ( function_exists( 'mcs_api_update_item' ) && false !== $item[0] ) {
			// This ID is the event ID, not the occur ID; get_item receives occur_id.
			if ( $update_id ) {
				$data = mcs_api_update_item( $request, $item, $params['id'], 'edit', $update_id );
			} else {
				$data = mcs_api_create_item( $request, $item, $params['id'] );
			}
			if ( is_array( $data ) ) {
				return new WP_REST_Response( $data, 200 );
			}
		}
		$error = ( '' === $item[3] ) ? __( 'Unable to create event. Invalid data submitted.', 'my-calendar-pro' ) : $item[3];

		return new WP_Error(
			'cant-update',
			$error,
			array( 'status' => 500 )
		);
	}

	/**
	 * Delete one item from the collection
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return WP_Error|WP_REST_Request
	 */
	public function delete_item( $request ) {
		// for deletion, all I need is an ID.
		$item = $this->prepare_item_for_delete( $request );

		if ( function_exists( 'mcs_api_delete_item' ) ) {
			if ( ! $item ) {
				// If $item is false this event doesn't exist in the target site. Return silently.
				return;
			} else {
				$deleted = mcs_api_delete_item( $item );
				if ( $deleted ) {
					return new WP_REST_Response( true, 200 );
				}
			}
		}

		return new WP_Error(
			'cant-delete',
			__( 'Unable to delete your event.', 'my-calendar-pro' ),
			array( 'status' => 500 )
		);
	}

	/**
	 * Check if a given request has access to get items
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return WP_Error|bool
	 */
	public function get_items_permissions_check( $request ) {
		// The collection request runs the same queries that My Calendar does internally, and any permissions are checked within those queries.

		return apply_filters( 'mc_api_items_permissions_check', true, $request );
	}

	/**
	 * Check if a given request has access to get a specific item
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return WP_Error|bool
	 */
	public function get_item_permissions_check( $request ) {
		// item permissions needs to verify that an individual item is public.
		return $this->get_items_permissions_check( $request );
	}

	/**
	 * Check if a given request has access to create items
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return WP_Error|bool
	 */
	public function create_item_permissions_check( $request ) {

		return mcs_api_authenticate( $request );
	}

	/**
	 * Check if a given request has access to update a specific item
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 *
	 * @return WP_Error|bool
	 */
	public function update_item_permissions_check( $request ) {

		return $this->create_item_permissions_check( $request );
	}

	/**
	 * Check if a given request has access to delete a specific item
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 * @return WP_Error|bool
	 */
	public function delete_item_permissions_check( $request ) {

		return $this->create_item_permissions_check( $request );
	}

	/**
	 * Prepare the item for create or update operation
	 *
	 * @param WP_REST_Request $request Request object.
	 *
	 * @return WP_Error|object $prepared_item
	 */
	protected function prepare_item_for_database( $request ) {
		// Run all event checking here.
		$event = $request->get_params();
		// Filter event data sent from remote site.
		$event = apply_filters( 'mcs_api_filter_input', $event, $request );

		$data   = $event['data'];
		$action = isset( $event['action'] ) ? $event['action'] : 'add';
		$nonce  = wp_create_nonce( 'event_nonce' );
		// Replace the nonce, as local site won't accept the remote site's nonce value.
		$data['event_nonce_name'] = $nonce;

		$data = mc_check_data( $action, $data, 0 );

		// Add category into resulting data.
		$data['category'] = $event['category'];

		return $data;
	}

	/**
	 * Prepare the item for delete operation
	 *
	 * @param WP_REST_Request $request Request object.
	 *
	 * @return WP_Error|object $prepared_item
	 */
	protected function prepare_item_for_delete( $request ) {
		// All I need is the ID; make sure it's in appropriate place.
		$params    = $request->get_params();
		$delete_id = mcs_api_check_requested_id( $params );
		$data      = ( $delete_id ) ? absint( $delete_id ) : false;

		return $data;
	}

	/**
	 * Prepare the item for the REST response
	 *
	 * @param mixed           $item WordPress representation of the item.
	 * @param WP_REST_Request $request Request object.
	 *
	 * @return mixed
	 */
	public function prepare_item_for_response( $item, $request ) {
		// Filter the item for the REST response.
		apply_filters( 'mcs_filter_rest_response', $item, $request );

		return $item;
	}

	/**
	 * Get the query params for collections
	 *
	 * @return array
	 */
	public function get_collection_params() {
		return array(
			'from'     => array(
				'description'       => 'Starting date.',
				'type'              => 'string',
				'default'           => gmdate( 'Y-m-d' ),
				'sanitize_callback' => 'sanitize_text_field',
			),
			'to'       => array(
				'description'       => 'Ending Date.',
				'type'              => 'string',
				'default'           => gmdate( 'Y-m-d', time() + 60 * 60 * 24 * 30 ),
				'sanitize_callback' => 'sanitize_text_field',
			),
			'category' => array(
				'description'       => 'Category to select from.',
				'type'              => 'integer',
				'sanitize_callback' => 'absint',
			),
			'ltype'    => array(
				'description'       => 'Type of location filter.',
				'type'              => 'string',
				'sanitize_callback' => 'sanitize_text_field',
			),
			'lvalue'   => array(
				'description'       => 'Location filtering value.',
				'type'              => 'string',
				'sanitize_callback' => 'sanitize_text_field',
			),
			'author'   => array(
				'description'       => 'Author ID.',
				'type'              => 'integer',
				'sanitize_callback' => 'absint',
			),
			'host'     => array(
				'description'       => 'Host ID.',
				'type'              => 'integer',
				'sanitize_callback' => 'absint',
			),
			'search'   => array(
				'description'       => 'Search Term',
				'type'              => 'string',
				'sanitize_callback' => 'sanitize_text_field',
			),
		);
	}
}
