<?php
/**
 * Payment management
 *
 * @category Features
 * @package  My Calendar Pro
 * @author   Joe Dolson
 * @license  GPLv3
 * @link     https://www.joedolson.com/my-calendar-pro/
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Print active payment filters.
 *
 * @param array $filters Array of filters applied.
 */
function mcs_payment_filters( $filters ) {
	if ( ! empty( $filters ) ) {
		?>
		<ul class="mcs-payment-filters">
		<?php
		foreach ( $filters as $filter ) {
			?>
			<li><?php echo wp_kses_post( $filter ); ?></li>
			<?php
		}
		?>
		</ul>
		<?php
	}
}

/**
 * Print payment list header rows.
 */
function mcs_payment_header_rows() {
	?>
	<tr>
		<th scope='col'><?php esc_html_e( 'ID', 'my-calendar-pro' ); ?></th>
		<th scope='col'><?php esc_html_e( 'Key', 'my-calendar-pro' ); ?></th>
		<th scope='col'><?php esc_html_e( 'Price', 'my-calendar-pro' ); ?></th>
		<th scope='col'><?php esc_html_e( 'Status', 'my-calendar-pro' ); ?></th>
		<th scope='col'><?php esc_html_e( 'Transaction', 'my-calendar-pro' ); ?></th>
		<th scope='col'><?php esc_html_e( 'Purchaser', 'my-calendar-pro' ); ?></th>
		<th scope='col'><?php esc_html_e( 'Uses', 'my-calendar-pro' ); ?></th>
	</tr>
	<?php
}

/**
 * Print the manual payment creation and editing form.
 */
function mcs_admin_payment_form() {
	?>
<div class="ui-sortable meta-box-sortables">
	<div class="postbox">
		<?php
		if ( isset( $_GET['mcs-payment'] ) ) {
			$payment_id = absint( $_GET['mcs-payment'] );
			$payment    = mcs_get_payment( $payment_id );
			// Translators: payment ID.
			$heading = sprintf( __( 'Editing payment %s', 'my-calendar-pro' ), '<code>' . $payment_id . '</code>' );
		} else {
			$heading = __( 'Add a manual payment', 'my-calendar-pro' );
		}
		?>
		<h2 class="heading"><?php echo wp_kses_post( $heading ); ?></h2>
		<div class="inside">
		<?php
		$quantity         = 1;
		$price            = get_option( 'mcs_submission_fee' );
		$first_name       = '';
		$last_name        = '';
		$email            = '';
		$transaction_date = date_i18n( 'j F, Y' );
		$action_url       = admin_url( 'admin.php?page=my-calendar-payments' );
		$total            = false;
		$status           = false;
		$submit_text      = __( 'Add Payment', 'my-calendar-pro' );
		$delete_button    = '';
		if ( isset( $_POST['mcs-delete-payment'] ) ) {
			$payment = absint( $_GET['mcs-payment'] );
			$nonce   = ( isset( $_POST['_wpnonce'] ) ) ? $_POST['_wpnonce'] : false;
			if ( wp_verify_nonce( $nonce, 'my-calendar-payments' ) ) {
				$response = mcs_delete_payment( $payment );
				wp_admin_notice(
					$response,
					array(
						'type' => 'success',
					)
				);
				unset( $_GET['mcs-payment'] );
			}
		}
		if ( isset( $_POST['mc-submit-payments'] ) ) {
			if ( isset( $_GET['mcs-payment'] ) ) {
				$post        = map_deep( $_POST, 'sanitize_text_field' );
				$response    = mcs_edit_payment( $post, $payment );
				$submit_text = __( 'Edit Payment', 'my-calendar-pro' );
			} else {
				$post     = map_deep( $_POST, 'sanitize_text_field' );
				$response = mcs_add_payment( $post );
			}
			if ( $response ) {
				wp_admin_notice(
					$response['message'],
					array(
						'type' => $response['type'],
					)
				);
			}

			$quantity         = isset( $post['quantity'] ) ? (int) $post['quantity'] : $quantity;
			$price            = isset( $post['price'] ) ? (float) $post['price'] : $price;
			$first_name       = isset( $post['first_name'] ) ? $post['first_name'] : $first_name;
			$last_name        = isset( $post['last_name'] ) ? $post['last_name'] : $last_name;
			$email            = isset( $post['email'] ) ? $post['email'] : $email;
			$transaction_date = isset( $post['transaction_date'] ) ? $post['transaction_date'] : $transaction_date;
		} else {
			if ( isset( $_GET['mcs-payment'] ) ) {
				$quantity         = $payment['quantity'];
				$total            = $payment['total'];
				$status           = $payment['status'];
				$price            = $payment['price'];
				$first_name       = $payment['first_name'];
				$last_name        = $payment['last_name'];
				$email            = $payment['payer_email'];
				$transaction_date = $payment['transaction_date'];
				$action_url       = add_query_arg( 'mcs-payment', $payment_id, admin_url( 'admin.php?page=my-calendar-payments' ) );
				$submit_text      = __( 'Edit Payment', 'my-calendar-pro' );
				$delete_button    = true;
			}
		}
		?>
		<form method="post" action="<?php echo esc_url( $action_url ); ?>" class='mcs-manage-payment'>
			<div>
			<input type="hidden" name="_wpnonce" value="<?php echo esc_attr( wp_create_nonce( 'my-calendar-payments' ) ); ?>" />
			</div>
			<ul>
				<li>
					<label for="quantity"><?php esc_html_e( 'Quantity', 'my-calendar-pro' ); ?></label> <input type="text" name="quantity" required id="quantity" size="6" value="<?php echo esc_attr( $quantity ); ?>" />
					<label for="price"><?php esc_html_e( 'Price Paid (total)', 'my-calendar-pro' ); ?></label> <input type="text" name="price" required id="price" size="6" value="<?php echo esc_attr( $price ); ?>" />
				</li>
				<?php
				if ( $total || isset( $payment['total'] ) ) {
					?>
				<li>
					<label for="total"><?php esc_html_e( 'Submissions Purchased', 'my-calendar-pro' ); ?></label> 
					<input type="number" required name="total" id="total" size="10" value="<?php echo esc_attr( $payment['total'] ); ?>" />
				</li>
					<?php
				}
				?>
				<li>
					<label for="first_name"><?php esc_html_e( 'First Name', 'my-calendar-pro' ); ?></label> 
					<input type="text" name="first_name" required id="first_name" class="widefat" value="<?php echo esc_attr( $first_name ); ?>" />
				</li>
				<li>
					<label for="last_name"><?php esc_html_e( 'Last Name', 'my-calendar-pro' ); ?></label> 
					<input type="text" name="last_name" required id="last_name" class="widefat" value="<?php echo esc_attr( $last_name ); ?>" />
				</li>
				<li>
					<label for="email"><?php esc_html_e( 'Email', 'my-calendar-pro' ); ?></label> 
					<input type="email" name="email" id="email" class="widefat" value="<?php echo esc_attr( $email ); ?>" required />
				</li>
				<li>
					<label for="transaction_date"><?php esc_html_e( 'Transaction Date', 'my-calendar-pro' ); ?></label> 
					<input type="date" name="transaction_date" id="transaction_date" size="20" value="<?php echo esc_attr( $transaction_date ); ?>" required />
				</li>
				<?php
				if ( $status || isset( $payment['status'] ) ) {
					$status = $payment['status'];
					?>
				<li>
					<label for="status"><?php esc_html_e( 'Status', 'my-calendar-pro' ); ?></label>
					<select name="status" id="status">
						<option value='Completed' <?php selected( 'Completed', $status, false ); ?>><?php esc_html_e( 'Completed', 'my-calendar-pro' ); ?></option>
						<option value='Pending' <?php selected( 'Pending', $status, false ); ?>><?php esc_html_e( 'Pending', 'my-calendar-pro' ); ?></option>
						<option value='Failed' <?php selected( 'Failed', $status, false ); ?>><?php esc_html_e( 'Failed', 'my-calendar-pro' ); ?></option>
						<option value='Refunded' <?php selected( 'Refunded', $status, false ); ?>><?php esc_html_e( 'Refunded', 'my-calendar-pro' ); ?></option>
					</select>
				</li>
					<?php
				}
				?>
			</ul>
			<p>
				<input type="submit" name="mc-submit-payments" class="button-primary" value="<?php echo esc_attr( $submit_text ); ?>" />
				<?php
				if ( $delete_button ) {
					?>
				<button name="mcs-delete-payment" class="mcs-delete button-secondary"><?php esc_html_e( 'Delete Payment', 'my-calendar-pro' ); ?></button>
					<?php
				}
				?>
			</p>
		</form>
		</div>
	</div>
</div>
	<?php
}

/**
 * Print the admin payment search form.
 */
function mcs_admin_payment_search_form() {
	?>
<div class="ui-sortable meta-box-sortables">
	<div class="postbox support">
		<h2 class="heading"><?php esc_html_e( 'Search Transactions', 'my-calendar-pro' ); ?></h2>
		<div class='inside'>
			<form method="post" action="<?php echo esc_url( admin_url( 'admin.php?page=my-calendar-payments' ) ); ?>">
				<div class="mcs-search columns">
					<?php
					$post   = map_deep( wp_unslash( $_POST ), 'sanitize_text_field' );
					$name   = ( isset( $post['name'] ) ) ? $post['name'] : '';
					$email  = ( isset( $post['email'] ) ) ? $post['email'] : '';
					$txn_id = ( isset( $post['txn_id'] ) ) ? $post['txn_id'] : '';
					$hash   = ( isset( $post['hash'] ) ) ? $post['hash'] : '';
					$before = ( isset( $post['before'] ) ) ? $post['before'] : '';
					$after  = ( isset( $post['after'] ) ) ? $post['after'] : '';
					$status = ( isset( $post['status'] ) ) ? $post['status'] : '';
					wp_nonce_field( 'mcs-search-payments' );

					echo "<div class='column-1'><p><label for='pname'>" . esc_html__( 'Name of Payer', 'my-calendar-pro' ) . "</label> <input type='text' id='pname' name='name' value='" . esc_attr( $name ) . "' /></p>";
					echo "<p><label for='pemail'>" . esc_html__( 'Email of Payer', 'my-calendar-pro' ) . "</label> <input type='text' id='pemail' name='email' value='" . esc_attr( $email ) . "' /></p>";
					echo "<p><label for='ptrans'>" . esc_html__( 'Transaction ID', 'my-calendar-pro' ) . "</label> <input type='text' id='ptrans' name='txn_id' value='" . esc_attr( $txn_id ) . "' /></p>";
					echo "<p><label for='lkey'>" . esc_html__( 'Payment Key', 'my-calendar-pro' ) . "</label> <input type='text' id='lkey' name='hash' value='" . esc_attr( $hash ) . "' /></p>";
					echo "</div><div class='column-2'><p><label for='adate'>" . esc_html__( 'After', 'my-calendar-pro' ) . "</label> <input type='date' id='adate' name='after' value='" . esc_attr( $after ) . "' /></p>";
					echo "<p><label for='bdate'>" . esc_html__( 'Before', 'my-calendar-pro' ) . "</label> <input type='date' id='bdate' name='before' value='" . esc_attr( $before ) . "' /></p>";
					echo "<p><label for='status'>" . esc_html__( 'Status of payment', 'my-calendar-pro' ) . "</label> <select name='status' id='status'>";
					echo "<option value=''>" . esc_html__( 'All statuses', 'my-calendar-pro' ) . '</option>';
					echo "<option value='Completed'" . selected( 'Completed', $status, false ) . '>' . esc_html__( 'Completed Payments', 'my-calendar-pro' ) . '</option>';
					echo "<option value='Pending'" . selected( 'Pending', $status, false ) . '>' . esc_html__( 'Pending Payments', 'my-calendar-pro' ) . '</option>';
					echo "<option value='Failed'" . selected( 'Failed', $status, false ) . '>' . esc_html__( 'Failed Payments', 'my-calendar-pro' ) . '</option>';
					echo "<option value='Refunded'" . selected( 'Refunded', $status, false ) . '>' . esc_html__( 'Refunded Payments', 'my-calendar-pro' ) . '</option>';
					echo '</select>';
					echo "</div><p class='submit'><button class='button-secondary'>" . esc_html__( 'Search Payments', 'my-calendar-pro' ) . '</button></p>';
					?>
				</div>
			</form>
		</div>
	</div>
</div>
	<?php
}

/**
 * Print the payments list.
 */
function mcs_admin_payments_list() {
	global $wpdb;
	?>
<div class="ui-sortable meta-box-sortables">
	<div class="postbox">
		<h2 class="heading"><?php esc_html_e( 'My Calendar Event Payments', 'my-calendar-pro' ); ?></h2>
		<div class="inside">
		<?php
		$name_query   = '';
		$email_query  = '';
		$txn_id_query = '';
		$hash_query   = '';
		$status_query = '';
		$date_query   = '';
		$filters      = array();
		$filtered     = false;

		$nonce = ( isset( $_POST['_wpnonce'] ) ) ? $_POST['_wpnonce'] : false;
		if ( $nonce && wp_verify_nonce( $_POST['_wpnonce'], 'mcs-search-payments' ) ) {
			$filtered = true;
			if ( ! empty( $_POST['name'] ) ) {
				$name_query = " AND (first_name LIKE '%" . esc_sql( $wpdb->esc_like( $_POST['name'] ) ) . "%' OR last_name LIKE '%" . esc_sql( $_POST['name'] ) . "%') ";
				// Translators: searched name.
				$filters[] = sprintf( __( 'Name includes "%s"', 'my-calendar-pro' ), esc_html( $_POST['name'] ) );
			}
			if ( ! empty( $_POST['email'] ) ) {
				$email_query = " AND payer_email LIKE '%" . esc_sql( $wpdb->esc_like( $_POST['email'] ) ) . "%' ";
				// Translators: searched email.
				$filters[] = sprintf( __( 'Email includes "%s"', 'my-calendar-pro' ), esc_html( $_POST['email'] ) );
			}
			if ( ! empty( $_POST['txn_id'] ) ) {
				$txn_id_query = " AND txn_id = '" . esc_sql( $_POST['txn_id'] ) . "' ";
				// Translators: searched transaction ID.
				$filters[] = sprintf( __( 'Transaction ID is "%s"', 'my-calendar-pro' ), esc_html( $_POST['txn_id'] ) );
			}
			if ( ! empty( $_POST['hash'] ) ) {
				$hash_query = " AND hash = '" . esc_sql( $_POST['hash'] ) . "' ";
				// Translators: searched payment key.
				$filters[] = sprintf( __( 'Payment key is "%s"', 'my-calendar-pro' ), esc_html( $_POST['hash'] ) );
			}
			if ( ! empty( $_POST['before'] ) || ! empty( $_POST['after'] ) ) {
				$before     = ( ! empty( $_POST['before'] ) ) ? $_POST['before'] : current_time( 'Y-m-d' );
				$after      = ( ! empty( $_POST['after'] ) ) ? $_POST['after'] : current_time( 'Y-m-d' );
				$date_query = " AND DATE(transaction_date) BETWEEN '$after 00:00:00' AND '$before 00:00:00' ";
				// Translators: 1) searched start date 2) searched end date.
				$filters[] = sprintf( __( 'Date is between "%1$s" and "%2$s"', 'my-calendar-pro' ), esc_html( $_POST['after'] ), esc_html( $_POST['before'] ) );
			}
			if ( ! empty( $_POST['status'] ) ) {
				$status_query = " AND status = '" . esc_sql( $_POST['status'] ) . "'";
				// Translators: searched status.
				$filters[] = sprintf( __( 'Payment status is "%s"', 'my-calendar-pro' ), esc_html( $_POST['status'] ) );
			}
		}
		$current        = empty( $_GET['paged'] ) ? 1 : intval( $_GET['paged'] );
		$items_per_page = 50;

		$sql         = 'SELECT * FROM ' . my_calendar_payments_table() . "
				WHERE TRUE AND item_number = 1
				$name_query
				$email_query
				$txn_id_query
				$hash_query
				$status_query
				$date_query
				ORDER BY id DESC";
		$sql_limited = $sql . ' LIMIT ' . ( ( $current - 1 ) * $items_per_page ) . ', ' . $items_per_page;

		$rows       = $wpdb->get_results( $sql_limited ); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
		$found_rows = $wpdb->get_col( str_replace( '*', 'COUNT(*)', $sql ) ); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
		$items      = $found_rows[0];

		if ( $items > 0 ) {
			$items_per_page = 20;
			$num_pages      = ceil( $items / $items_per_page );
			$page_links     = paginate_links(
				array(
					'base'      => add_query_arg( 'paged', '%#%' ),
					'format'    => '',
					'prev_text' => __( '&larr; Previous', 'my-calendar-pro' ),
					'next_text' => __( 'Next &rarr;', 'my-calendar-pro' ),
					'total'     => $num_pages,
					'current'   => $current,
				)
			);
			mcs_payment_filters( $filters );
			if ( $num_pages > 1 ) {
				?>
				<nav class='tablenav' aria-label='<?php esc_attr_e( 'Payments pages', 'my-calendar-pro' ); ?>'>
					<div class='tablenav-pages'>
						<?php echo wp_kses_post( $page_links ); ?>
					</div>
				</nav>
				<?php
			}
			?>
			<table class='widefat wp-list-table striped'>
				<thead>
					<?php mcs_payment_header_rows(); ?>
				</thead>
				<tbody>
			<?php
			$action_url = admin_url( 'admin.php?page=my-calendar-payments' );
			$total      = 0.00;
			$pending    = 0.00;
			foreach ( $rows as $row ) {
				?>
					<tr>
						<th scope="row">
							<a href="<?php echo esc_url( add_query_arg( 'mcs-payment', $row->id, $action_url ) ); ?>"><?php echo esc_html( $row->id ); ?></a>
						</td>
						<td>
							<?php echo esc_html( $row->hash ); ?>
							<div class="row-actions">
								<a href="<?php echo esc_url( add_query_arg( 'mcs_receipt', $row->hash, home_url() ) ); ?>"><?php esc_html_e( 'View Receipt', 'my-calendar-pro' ); ?></a>
							</div>
						</td>
						<td><?php echo esc_html( mcs_money_format( $row->price ) ); ?></td>
						<td><?php echo esc_html( $row->status ); ?></td>
						<td><?php echo esc_html( $row->transaction_date ); ?><br /><?php echo esc_html( $row->txn_id ); ?></td>
						<td>
							<a href="mailto:<?php echo esc_html( $row->payer_email ); ?>">
							<?php echo esc_html( wp_unslash( $row->first_name . ' ' . wp_unslash( $row->last_name ) ) ); ?></a><br />
							<?php echo esc_html( $row->payer_email ); ?>
						</td>
						<td><?php echo esc_html( $row->quantity . '/' . $row->total ); ?></td>
					</tr>
				<?php
				$total   = ( 'Completed' === $row->status ) ? $total + $row->price : $total;
				$pending = ( 'Pending' === $row->status ) ? $pending + $row->price : $pending;
			}
			?>
					</tbody>
				<tfoot>
					<?php mcs_payment_header_rows(); ?>
				</tfoot>
			</table>
			<?php
			// Translators: Number of transactions found.
			$report = '<strong>' . sprintf( _n( '%s transaction', '%s transactions', $items, 'my-calendar-pro' ), $items ) . '</strong> &bull; ';
			// Translators: 1) Amount paid, 2) Amount pending.
			$report .= sprintf( __( 'Total Paid: %1$s. Pending Payments: %2$s', 'my-calendar-pro' ), mcs_money_format( $total ), mcs_money_format( $pending ) );
			echo wp_kses_post( "<p>$report</p>" );
		} else {
			if ( $filtered ) {
				mcs_payment_filters( $filters );
				esc_html_e( 'There are no payments meeting those search criteria.', 'my-calendar-pro' );
			} else {
				esc_html_e( 'You have not yet received any payments for event submissions', 'my-calendar-pro' );
			}
		}
		?>
		</div>
	</div>
</div>
	<?php
}

/**
 * Generate sales tracking & entry page.
 */
function mcs_sales_page() {
	mcs_check();
	?>
<div class='wrap my-calendar-admin my-calendar-pro-payments'>
	<h1><?php esc_html_e( 'My Calendar Pro - Event Submission Sales', 'my-calendar-pro' ); ?></h1>
	<div class="postbox-container jcd-wide mcs-payment-page">
		<div class="metabox-holder">
			<?php mcs_admin_payment_form(); ?>
			<?php mcs_admin_payments_list(); ?>
			<?php mcs_admin_payment_search_form(); ?>
		</div>
	</div>
	<div class="postbox-container jcd-narrow">
		<div class="metabox-holder">
			<?php mcs_admin_earnings_summary(); ?>
		</div>
	</div>
</div>
	<?php
}

/**
 * Print the earnings summary.
 */
function mcs_admin_earnings_summary() {
	global $wpdb;
	?>
<div class="ui-sortable meta-box-sortables">
	<div class="postbox support">
		<h2 class="heading"><?php esc_html_e( 'Earnings Summary', 'my-calendar-pro' ); ?></h2>
		<div class='inside'>
			<div class="mcs-earnings">
				<h3><?php esc_html_e( 'Latest Earnings', 'my-calendar-pro' ); ?></h3>
			<?php
			// Current Month.
			$earnings = 'SELECT SUM(price) FROM ' . my_calendar_payments_table() . " WHERE MONTH(transaction_date) = '" . gmdate( 'n' ) . "' AND YEAR(transaction_date) = '" . gmdate( 'Y' ) . "' AND item_number = 1 AND status = 'Completed'";
			$sum      = ( $wpdb->get_var( $earnings ) ) ? $wpdb->get_var( $earnings ) : '0.00'; // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
			// Translators: total amount of earnings.
			echo '<p>' . sprintf( esc_html__( 'Current month: %s', 'my-calendar-pro' ), esc_html( mcs_money_format( $sum ) ) ) . '<br />';

			// Current Year.
			$earnings = 'SELECT SUM(price) FROM ' . my_calendar_payments_table() . " WHERE YEAR(transaction_date) = '" . gmdate( 'Y' ) . "' AND item_number = 1 AND status = 'Completed'";
			$sum      = ( $wpdb->get_var( $earnings ) ) ? $wpdb->get_var( $earnings ) : '0.00'; // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
			// Translators: total amount of earnings.
			echo sprintf( esc_html__( 'Current year: %s', 'my-calendar-pro' ), esc_html( mcs_money_format( $sum ) ) ) . '</p>';
			echo '<h3>' . esc_html__( 'Previous Earnings', 'my-calendar-pro' ) . '</h3>';

			// Last Month.
			$earnings = 'SELECT SUM(price) FROM ' . my_calendar_payments_table() . " WHERE MONTH(transaction_date) = '" . gmdate( 'n', strtotime( '-1 month' ) ) . "' AND YEAR(transaction_date) = '" . gmdate( 'Y', strtotime( '-1 month' ) ) . "' AND item_number = 1 AND status = 'Completed'";
			$sum      = ( $wpdb->get_var( $earnings ) ) ? $wpdb->get_var( $earnings ) : '0.00'; // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
			// Translators: total amount of earnings.
			echo '<p>' . sprintf( esc_html__( 'Last month: %s', 'my-calendar-pro' ), esc_html( mcs_money_format( $sum ) ) ) . '<br />';

			// Last Year.
			$earnings = 'SELECT SUM(price) FROM ' . my_calendar_payments_table() . " WHERE YEAR(transaction_date) = '" . gmdate( 'Y', strtotime( '-1 year' ) ) . "' AND item_number = 1 AND status = 'Completed'";
			$sum      = ( $wpdb->get_var( $earnings ) ) ? $wpdb->get_var( $earnings ) : '0.00'; // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
			// Translators: total amount of earnings.
			echo sprintf( esc_html__( 'Last year: %s', 'my-calendar-pro' ), esc_html( mcs_money_format( $sum ) ) ) . '</p>';

			echo '<h3>' . esc_html__( 'All time', 'my-calendar-pro' ) . '</h3>';
			// Total all time.
			$earnings = 'SELECT SUM(price) FROM ' . my_calendar_payments_table() . " WHERE item_number = 1 AND status = 'Completed'";
			$sum      = ( $wpdb->get_var( $earnings ) ) ? $wpdb->get_var( $earnings ) : '0.00'; // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
			// Translators: total amount of earnings.
			echo '<p>' . sprintf( esc_html__( 'Total: %s', 'my-calendar-pro' ), esc_html( mcs_money_format( $sum ) ) ) . '</p>';
			?>
			</div>
		</div>
	</div>
</div>
	<?php
}

/**
 * Apply money formatting.
 *
 * @param float|integer $price Price.
 *
 * @return string
 */
function mcs_money_format( $price ) {
	if ( is_numeric( $price ) ) {
		$symbol        = mcs_symbols( get_option( 'mcs_currency' ) );
		$dec_point     = '.';
		$thousands_sep = ',';

		return $symbol . '<span class="price">' . number_format( $price, 2, $dec_point, $thousands_sep ) . '</span>';
	} else {
		return '';
	}
}


/**
 * Return current currency symbol.
 *
 * @param array $currency Currencies.
 *
 * @return string
 */
function mcs_symbols( $currency ) {
	if ( ! $currency ) {
		return '$';
	}
	$currencies = mcs_currency();
	if ( ! isset( $currencies[ $currency ] ) ) {
		return $currency;
	}

	return $currencies[ $currency ]['symbol'];
}

/**
 * All currencies.
 *
 * @return array
 */
function mcs_currency() {
	$currencies = apply_filters(
		'mcs_currencies',
		array(
			'USD' => array(
				'symbol'      => '$',
				'description' => __( 'U.S. Dollars ($)', 'my-calendar-pro' ),
			),
			'EUR' => array(
				'symbol'      => '€',
				'description' => __( 'Euros (€)', 'my-calendar-pro' ),
			),
			'AUD' => array(
				'symbol'      => 'A $',
				'description' => __( 'Australian Dollars (A $)', 'my-calendar-pro' ),
			),
			'CAD' => array(
				'symbol'      => 'C $',
				'description' => __( 'Canadian Dollars (C $)', 'my-calendar-pro' ),
			),
			'GBP' => array(
				'symbol'      => '£',
				'description' => __( 'Pounds Sterling (£)', 'my-calendar-pro' ),
			),
			'INR' => array(
				'symbol'      => '₹',
				'description' => __( 'Indian Rupees (₹)', 'my-calendar-pro' ),
			),
			'JPY' => array(
				'symbol'      => '¥',
				'description' => __( 'Yen (¥)', 'my-calendar-pro' ),
				'zerodecimal' => true,
			),
			'NZD' => array(
				'symbol'      => '$',
				'description' => __( 'New Zealand Dollar ($)', 'my-calendar-pro' ),
			),
			'CHF' => array(
				'symbol'      => 'CHF ',
				'description' => __( 'Swiss Franc', 'my-calendar-pro' ),
			),
			'HKD' => array(
				'symbol'      => '$',
				'description' => __( 'Hong Kong Dollar ($)', 'my-calendar-pro' ),
			),
			'SGD' => array(
				'symbol'      => '$',
				'description' => __( 'Singapore Dollar ($)', 'my-calendar-pro' ),
			),
			'SEK' => array(
				'symbol'      => 'kr ',
				'description' => __( 'Swedish Krona', 'my-calendar-pro' ),
			),
			'DKK' => array(
				'symbol'      => 'kr ',
				'description' => __( 'Danish Krone', 'my-calendar-pro' ),
			),
			'PLN' => array(
				'symbol'      => 'zł',
				'description' => __( 'Polish Zloty', 'my-calendar-pro' ),
			), // this is triggering an error. Why.
			'NOK' => array(
				'symbol'      => 'kr ',
				'description' => __( 'Norwegian Krone', 'my-calendar-pro' ),
			),
			'HUF' => array(
				'symbol'      => 'Ft ',
				'description' => __( 'Hungarian Forint', 'my-calendar-pro' ),
				'zerodecimal' => true,
			),
			'ILS' => array(
				'symbol'      => '₪',
				'description' => __( 'Israeli Shekel', 'my-calendar-pro' ),
			),
			'MXN' => array(
				'symbol'      => '$',
				'description' => __( 'Mexican Peso', 'my-calendar-pro' ),
			),
			'BRL' => array(
				'symbol'      => 'R$',
				'description' => __( 'Brazilian Real', 'my-calendar-pro' ),
			),
			'MYR' => array(
				'symbol'      => 'RM',
				'description' => __( 'Malaysian Ringgits', 'my-calendar-pro' ),
			),
			'PHP' => array(
				'symbol'      => '₱',
				'description' => __( 'Philippine Pesos', 'my-calendar-pro' ),
			),
			'TWD' => array(
				'symbol'      => 'NT$',
				'description' => __( 'Taiwan New Dollars', 'my-calendar-pro' ),
				'zerodecimal' => true,
			),
			'THB' => array(
				'symbol'      => '฿',
				'description' => __( 'Thai Baht', 'my-calendar-pro' ),
			),
			'TRY' => array(
				'symbol'      => 'TRY ',
				'description' => __( 'Turkish Lira', 'my-calendar-pro' ),
			),
			'ZAR' => array(
				'symbol'      => 'R',
				'description' => __( 'South African Rand', 'my-calendar-pro' ),
			),
		)
	);
	ksort( $currencies );

	return $currencies;
}

/**
 * Add a manual payment.
 *
 * @param array $post POST data.
 *
 * @return void|array [message,type]
 */
function mcs_add_payment( $post ) {
	if ( isset( $post['mc-submit-payments'] ) ) {
		$nonce = $post['_wpnonce'];
		// Don't add if email is missing.
		if ( ! wp_verify_nonce( $nonce, 'my-calendar-payments' ) || ( empty( $post['email'] ) || ! is_email( $post['email'] ) ) ) {
			return array(
				'message' => __( 'Required fields not completed. All fields are required.', 'my-calendar-pro' ),
				'type'    => 'error',
			);
		}
		$args = mcs_create_payment( $post );

		if ( isset( $args['status'] ) && 'Completed' === $args['status'] ) {
			mcs_send_notifications( $args );
			return array(
				'message' => __( 'New Payment Added', 'my-calendar-pro' ),
				'type'    => 'success',
			);
		} else {
			return array(
				'message' => __( 'New Payment was not added.', 'my-calendar-pro' ),
				'type'    => 'error',
			);
		}
	}

	return false;
}

add_action( 'admin_enqueue_scripts', 'mcs_payment_enqueue_scripts' );
/**
 * Enqueue payment scripts.
 */
function mcs_payment_enqueue_scripts() {
	global $mcs_version;
	if ( SCRIPT_DEBUG && true === SCRIPT_DEBUG ) {
		$mcs_version = $mcs_version . uniqid();
	}
	if ( isset( $_GET['page'] ) && 'my-calendar-payments' === $_GET['page'] ) {
		wp_enqueue_script( 'mcs.payments', plugins_url( 'js/payments.js', __FILE__ ), array( 'jquery' ), $mcs_version, true );
	}
}

/**
 * Edit a payment from the admin.
 *
 * @param array $post POST data.
 * @param array $payment Payment data.
 *
 * @return void|array message
 */
function mcs_edit_payment( $post, $payment ) {
	if ( isset( $post['mc-submit-payments'] ) ) {
		$nonce = $post['_wpnonce'];
		// Don't add if email is missing.
		if ( ! wp_verify_nonce( $nonce, 'my-calendar-payments' ) || ( empty( $post['email'] ) || ! is_email( $post['email'] ) ) ) {
			return array(
				'message' => __( 'Required fields not completed. All fields are required.', 'my-calendar-pro' ),
				'type'    => 'error',
			);
		}
		$post = array_merge(
			$post,
			array(
				'item_number' => $payment['id'],
			)
		);
		$args = mcs_update_payment( $post, $payment );

		if ( isset( $args['status'] ) && 'Completed' === $args['status'] && 'Completed' !== $payment['status'] ) {
			mcs_send_notifications( $args );
		}
		return array(
			'message' => __( 'Payment edited.', 'my-calendar-pro' ),
			'type'    => 'success',
		);
	}

	return false;
}


/**
 * Create a new payment.
 *
 * @param array $post Posted data.
 *
 * @return array $args.
 */
function mcs_create_payment( $post ) {
	global $wpdb;
	// Price & Quantity minimum values available.
	$quantity         = ( isset( $post['quantity'] ) ) ? (int) $post['quantity'] : 1;
	$price            = sprintf( '%01.2f', $post['price'] );
	$first_name       = ( isset( $post['first_name'] ) ) ? $post['first_name'] : '';
	$last_name        = ( isset( $post['last_name'] ) ) ? $post['last_name'] : '';
	$email            = ( isset( $post['email'] ) ) ? is_email( $post['email'] ) : '';
	$transaction_date = ( isset( $post['transaction_date'] ) ) ? gmdate( 'Y-m-d h:m:s', strtotime( $post['transaction_date'] ) ) : current_time( 'Y-m-d h:m:s' );
	$txn_id           = ( isset( $post['txn_id'] ) ) ? $post['txn_id'] : 'Manual Entry';
	$status           = ( isset( $post['status'] ) ) ? $post['status'] : 'Completed';
	$uniqid           = uniqid( 'E' );
	$hash             = mcs_uniqid( $uniqid );
	$add              = array(
		'item_number'      => 1,
		'quantity'         => $quantity,
		'total'            => $quantity,
		'hash'             => $hash,
		'txn_id'           => $txn_id,
		'price'            => $price,
		'fee'              => '0.00',
		'status'           => $status,
		'transaction_date' => $transaction_date,
		'first_name'       => $first_name,
		'last_name'        => $last_name,
		'payer_email'      => $email,
	);

	$formats = array( '%d', '%d', '%d', '%s', '%s', '%f', '%f', '%s', '%s', '%s', '%s', '%s' );
	$insert  = $wpdb->insert( my_calendar_payments_table(), $add, $formats );
	$id      = $wpdb->insert_id;
	$args    = array(
		'first'    => $first_name,
		'last'     => $last_name,
		'email'    => $email,
		'price'    => $price,
		'hash'     => $hash,
		'quantity' => $quantity,
		'id'       => $id,
		'status'   => $status,
	);

	return $args;
}

/**
 * Register My Calendar Pro hooks.
 */
function mcs_register_hooks() {
	add_filter( 'mcs_payment_required', 'mcs_role_payment_required', 10, 1 );
	add_action( 'mcs_payment_settings_update', 'mcs_payment_settings_update', 10, 1 );
	add_filter( 'mcs_get_price', 'mcs_get_custom_price', 10, 1 );
	add_filter( 'mcs_submission_fee_settings', 'mcs_submission_fee_settings', 10, 1 );
}
add_action( 'init', 'mcs_register_hooks' );

/**
 * Check custom array for price of current logged_in role. If free for this role, treat as if payment not required.
 *
 * @param boolean $required Is payment required right now.
 *
 * @return boolean
 */
function mcs_role_payment_required( $required ) {
	if ( ! is_user_logged_in() ) {
		return $required;
	}
	$role      = get_current_user_role();
	$test_mode = get_option( 'mcs_use_sandbox' );
	// If in test mode, show payment form for admins.
	if ( ( 'administrator' === $role || 'super_admin' === $role ) && 'true' === $test_mode ) {
		// True means payment is required.
		return true;
	} elseif ( ( 'administrator' === $role || 'super_admin' === $role ) ) {
		return false;
	}
	if ( ( 'true' === get_option( 'mcs_payments' ) && ! is_user_logged_in() ) || ( is_user_logged_in() && ( mcs_get_price() > 0 ) ) ) {
		$return = true;
	} else {
		$return = false;
	}

	return $return;
}


/**
 * Save settings array of submission fees for different roles.
 *
 * @param array $post POST data.
 */
function mcs_payment_settings_update( $post ) {
	$mcs_submission_fee = array_map( 'mcs_map_fee', $post['mcs_submission_fees'] );
	update_option( 'mcs_submission_fees', $mcs_submission_fee );
}

/**
 * Get price by role.
 *
 * @param boolean|float $price Price when known.
 * @param string        $role Role to check.
 *
 * @return mixed float|boolean $price.
 */
function mcs_custom_price( $price, $role = 'auto' ) {
	// get $price by role.
	if ( 'auto' === $role ) {
		$role = get_current_user_role();
	}
	$prices = get_option( 'mcs_submission_fees' );
	$price  = ( isset( $prices[ $role ] ) ) ? $prices[ $role ] : $price;

	return $price;
}


/**
 * Get custom price based on role.
 *
 * @param float  $price Base price.
 * @param string $role User's role.
 *
 * @return float New price.
 */
function mcs_get_custom_price( $price = false, $role = 'auto' ) {
	$price = mcs_custom_price( $price, $role );

	return $price;
}

/**
 * Add fields for setting fees.
 *
 * @param string $html_output Default return value to append form to.
 *
 * @return string
 */
function mcs_submission_fee_settings( $html_output ) {
	// input HTML for roles - use mc tabs.
	$roles              = mcs_get_roles();
	$mcs_submission_fee = (array) get_option( 'mcs_submission_fees' );
	$default_fee        = get_option( 'mcs_submission_fee' );
	$html_output       .= ( 'true' === get_option( 'mcs_payments' ) ) ? '<fieldset><legend>' . __( 'User role-based pricing', 'my-calendar-pro' ) . '</legend><ul class="checkboxes">' : '';
	foreach ( $roles as $role => $rolename ) {
		$fee      = ( isset( $mcs_submission_fee[ $role ] ) ) ? $mcs_submission_fee[ $role ] : $default_fee;
		$rolename = esc_html( $rolename );
		$role     = esc_attr( $role );
		if ( 'true' === get_option( 'mcs_payments' ) ) {
			$html_output .= "<li><label for='mcs_submission_fee_$role'>Price: <strong>$rolename</strong></label> <input type='text' name='mcs_submission_fees[$role]' id='mcs_submission_fee_$role' value='" . esc_attr( $fee ) . "' /></li>";
		} else {
			$html_output .= "<input type='hidden' name='mcs_submission_fees[$role]' value='" . esc_attr( $fee ) . "' />";
		}
	}
	$html_output .= ( 'true' === get_option( 'mcs_payments' ) ) ? '</ul></fieldset>' : '';

	return $html_output;
}

/**
 * Get all valid roles & strip out admin and super admin.
 *
 * @return array
 */
function mcs_get_roles() {
	global $wp_roles;
	$roles = $wp_roles->get_names();
	unset( $roles['administrator'] );
	unset( $roles['super_admin'] );

	return $roles;
}

/**
 * Format the fee into money.
 *
 * @param float $fee Cost.
 *
 * @return string
 */
function mcs_map_fee( $fee ) {
	return sprintf( '%01.2f', $fee );
}

/**
 * Returns the role of the current user. If that user has
 * no role for the current blog, it returns false.
 *
 * @return string The name of the current role
 **/
function get_current_user_role() {
	global $wp_roles;
	$current_user = wp_get_current_user();
	$roles        = $current_user->roles;
	$role         = array_shift( $roles );

	return isset( $wp_roles->role_names[ $role ] ) ? $role : false;
}


/**
 * ID the My Calendar Payments Table
 */
function my_calendar_payments_table() {
	$option = (int) get_site_option( 'mc_multisite' );
	$choice = (int) get_option( 'mc_current_table' );
	switch ( $option ) {
		case 0:
			return MY_CALENDAR_PAYMENTS_TABLE;
			break;
		case 1:
			return MY_CALENDAR_GLOBAL_PAYMENTS_TABLE;
			break;
		case 2:
			return ( 1 === $choice ) ? MY_CALENDAR_GLOBAL_PAYMENTS_TABLE : MY_CALENDAR_PAYMENTS_TABLE;
			break;
		default:
			return MY_CALENDAR_PAYMENTS_TABLE;
	}
}
