<?php
/**
 * REST API Server UI
 *
 * @category API
 * @package  My Calendar Pro
 * @author   Joe Dolson
 * @license  GPLv3
 * @link     https://www.joedolson.com/my-calendar-pro/
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

require_once __DIR__ . '/classes/class-mcs-api-widget.php';
// Mobile UI for selecting which calendars/categories to view
// What to show by default?
// Concept: accordion display of calendars; accordion shows categories for that calendar & option to hide calendar.
// All hiding done via JS.

/**
 * Filter events based on saved _mc_event_source value (use as class to pick via JS
 * - Be able to sub select by combined category & source values
 * - UI Admin will have sources from mc_api_sources(); is whitelist & checks to indicate "show in front"
 *
 * @param array  $classes Classes to show on event.
 * @param object $event Event object.
 * @param string $uid Unique ID for event front end.
 * @param string $type Type of display.
 *
 * @return array Add source name class.
 */
function mcs_api_view_classes( $classes, $event, $uid, $type ) {
	$post   = $event->event_post;
	$source = get_post_meta( $post, '_mc_api_source', true );
	if ( $source && mcs_api_id_source( $source ) ) {
		$classes[] = sanitize_title( mcs_api_id_source( $source ) );
		$classes[] = sanitize_title( str_replace( array( 'http://', 'https://' ), '', $source ) );
	}

	return $classes;
}
add_filter( 'mc_event_classes', 'mcs_api_view_classes', 10, 4 );

/**
 * Take source URL and verify that it is a permitted source & get title.
 *
 * @param string $source URL of API source.
 *
 * @return Label of valid source.
 */
function mcs_api_id_source( $source ) {
	$sources = mcs_api_sources();
	// sources should be considered identical whether they're https or http, slashed or unslashed.
	// this way, if a source is changed from one to the other, references aren't lost.
	$source1 = str_replace( 'http:', 'https:', $source );
	$source2 = str_replace( 'https:', 'http:', $source );
	$source3 = trailingslashit( $source1 );
	$source4 = untrailingslashit( $source1 );
	$source5 = trailingslashit( $source2 );
	$source6 = untrailingslashit( $source2 );
	$id      = ( isset( $sources[ $source1 ] ) ) ? $sources[ $source1 ] : false;

	if ( ! $id ) {
		$id = ( isset( $sources[ $source2 ] ) ) ? $sources[ $source2 ] : false;
	}

	if ( ! $id ) {
		$id = ( isset( $sources[ $source3 ] ) ) ? $sources[ $source3 ] : false;
	}

	if ( ! $id ) {
		$id = ( isset( $sources[ $source4 ] ) ) ? $sources[ $source4 ] : false;
	}

	if ( ! $id ) {
		$id = ( isset( $sources[ $source5 ] ) ) ? $sources[ $source5 ] : false;
	}

	if ( ! $id ) {
		$id = ( isset( $sources[ $source6 ] ) ) ? $sources[ $source6 ] : false;
	}
	$display = $id['display'];

	if ( $id ) {
		$label = ( $display ) ? $id['label'] : 'disabled-source';
	} else {
		$label = false;
	}

	return $label;
}

/**
 * Display list of Sources
 *
 * @param string $current Current source.
 *
 * @return list of sources.
 */
function mcs_api_list_sources( $current = false ) {
	$sources = mcs_api_sources();
	$list    = '';
	foreach ( $sources as $source ) {
		$id       = sanitize_title( $source['label'] );
		$label    = esc_html( $source['label'] );
		$expanded = ( $current === $id ) ? 'true' : 'false';
		$display  = $source['display'];

		if ( $display ) {
			$list .= "<li><button type='button' data-controls='$id' aria-expanded='$expanded'><span class='dashicons dashicons-minus' aria-hidden='true'></span></span>$label</button></li>";
		}
	}

	if ( '' !== $list ) {
		return "<div class='mc-api-controls'><ul>$list</ul></div>";
	}
}

/**
 * Shortcode for listing source.
 *
 * @return string
 */
function mcs_list_api_sources() {

	return mcs_api_list_sources();
}
add_shortcode( 'my_calendar_api_sources', 'mcs_list_api_sources' );

/**
 * Enqueue scripts.
 */
function mcs_api_enqueue_scripts() {
	global $mcs_version;
	wp_enqueue_script( 'mcapi.filter', plugins_url( 'js/mcapi.filter.js', __FILE__ ), array( 'jquery' ), $mcs_version, true );
	wp_enqueue_style( 'mcapi.style', plugins_url( 'css/filters.css', __FILE__ ), array(), $mcs_version );
}
add_action( 'wp_enqueue_scripts', 'mcs_api_enqueue_scripts' );

/**
 * Add widget API controls.
 */
function mcs_api_register_widgets() {
	register_widget( 'Mcs_API_Widget' );
}
add_action( 'widgets_init', 'mcs_api_register_widgets' );
