<?php
/**
 * Notifications
 *
 * @category Settings
 * @package  My Calendar Pro
 * @author   Joe Dolson
 * @license  GPLv3
 * @link     https://www.joedolson.com/my-calendar-pro/
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

add_filter( 'mcs_custom_settings_update', 'mcs_update_notifications', 10, 2 );
/**
 * Save notification settings.
 *
 * @param string $updated Return value. Default false.
 * @param array  $post POST array.
 *
 * @return string
 */
function mcs_update_notifications( $updated, $post ) {
	// save settings.
	if ( isset( $post['notifications_settings'] ) ) {
		$mcs_from                            = is_email( $post['mcs_from'] ); // send from.
		$mcs_to                              = sanitize_text_field( $post['mcs_to'] ); // send to.
		$mcs_response                        = wp_kses_post( $post['mcs_response'] ); // admin email after submission.
		$mcs_edit_response                   = wp_kses_post( $post['mcs_edit_response'] ); // admin email after submission.
		$mcs_confirmation                    = wp_kses_post( $post['mcs_confirmation'] ); // submitter email after submission.
		$mcs_edit_confirmation               = wp_kses_post( $post['mcs_edit_confirmation'] ); // submitter email after submission.
		$mcs_subject                         = sanitize_text_field( $post['mcs_subject'] ); // subject line.
		$mcs_edit_subject                    = sanitize_text_field( $post['mcs_edit_subject'] ); // editing subject line.
		$mcs_confirmation_subject            = sanitize_text_field( $post['mcs_confirmation_subject'] );
		$mcs_edit_confirmation_subject       = sanitize_text_field( $post['mcs_edit_confirmation_subject'] );
		$mcs_publication_subject             = sanitize_text_field( $post['mcs_publication_subject'] );
		$mcs_publication                     = wp_kses_post( $post['mcs_publication'] );
		$mcs_html_email                      = ( isset( $post['mcs_html_email'] ) ) ? 'true' : 'false'; // send as HTML.
		$mcs_dont_send_submitter_email       = ( isset( $post['mcs_dont_send_submitter_email'] ) ) ? 'true' : 'false';
		$mcs_dont_send_admin_email           = ( isset( $post['mcs_dont_send_admin_email'] ) ) ? 'true' : 'false';
		$mcs_disable_edit_notifications      = ( isset( $post['mcs_disable_edit_notifications'] ) ) ? 'true' : 'false';
		$mcs_disable_user_edit_notifications = ( isset( $post['mcs_disable_user_edit_notifications'] ) ) ? 'true' : 'false';

		update_option( 'mcs_response', $mcs_response );
		update_option( 'mcs_edit_response', $mcs_edit_response );
		update_option( 'mcs_confirmation', $mcs_confirmation );
		update_option( 'mcs_edit_confirmation', $mcs_edit_confirmation );
		update_option( 'mcs_subject', $mcs_subject );
		update_option( 'mcs_edit_subject', $mcs_edit_subject );
		update_option( 'mcs_disable_edit_notifications', $mcs_disable_edit_notifications );
		update_option( 'mcs_disable_user_edit_notifications', $mcs_disable_user_edit_notifications );
		update_option( 'mcs_confirmation_subject', $mcs_confirmation_subject );
		update_option( 'mcs_edit_confirmation_subject', $mcs_edit_confirmation_subject );
		update_option( 'mcs_publication_subject', $mcs_publication_subject );
		update_option( 'mcs_publication', $mcs_publication );
		update_option( 'mcs_to', $mcs_to );
		update_option( 'mcs_from', $mcs_from );
		update_option( 'mcs_html_email', $mcs_html_email );
		update_option( 'mcs_dont_send_submitter_email', $mcs_dont_send_submitter_email );
		update_option( 'mcs_dont_send_admin_email', $mcs_dont_send_admin_email );
		$updated = __( 'My Calendar Notification settings updated.', 'my-calendar-pro' );
	}

	return $updated;
}

add_filter( 'mcs_settings_tabs', 'mcs_notifications_tabs' );
/**
 * Set up notifications tab.
 *
 * @param array $tabs all tabs.
 *
 * @return array tabs
 */
function mcs_notifications_tabs( $tabs ) {
	$tabs['notifications'] = __( 'Notifications', 'my-calendar-pro' );

	return $tabs;
}

add_filter( 'mcs_settings_panels', 'mcs_notifications_settings' );
/**
 * Set up notifications options panel.
 *
 * @param array $panels All panels.
 *
 * @return array
 */
function mcs_notifications_settings( $panels ) {
	$options                             = mcs_get_default_settings();
	$defaults                            = $options['defaults'];
	$mcs_to                              = get_option( 'mcs_to', get_option( 'admin_email' ) ); // send to.
	$mcs_from                            = get_option( 'mcs_from', get_option( 'admin_email' ) ); // send from.
	$mcs_subject                         = ( get_option( 'mcs_subject' ) ) ? get_option( 'mcs_subject' ) : $defaults['mcs_subject']; // subject line.
	$mcs_edit_subject                    = ( get_option( 'mcs_edit_subject' ) ) ? get_option( 'mcs_edit_subject' ) : $defaults['mcs_edit_subject']; // subject line.
	$mcs_disable_edit_notifications      = ( get_option( 'mcs_disable_edit_notifications' ) ) ? 'true' : 'false';
	$mcs_disable_user_edit_notifications = ( get_option( 'mcs_disable_user_edit_notifications' ) ) ? 'true' : 'false';
	$mcs_response                        = ( get_option( 'mcs_response' ) ) ? get_option( 'mcs_response' ) : $defaults['mcs_response']; // admin email after submission.
	$mcs_edit_response                   = ( get_option( 'mcs_edit_response' ) ) ? get_option( 'mcs_edit_response' ) : $defaults['mcs_edit_response']; // admin email after submission (edits).
	$mcs_confirmation                    = ( get_option( 'mcs_confirmation' ) ) ? get_option( 'mcs_confirmation' ) : $defaults['mcs_confirmation']; // submitter email after submission.
	$mcs_edit_confirmation               = ( get_option( 'mcs_edit_confirmation' ) ) ? get_option( 'mcs_edit_confirmation' ) : $defaults['mcs_edit_confirmation']; // submitter email after submission.
	$mcs_confirmation_subject            = ( get_option( 'mcs_confirmation_subject' ) ) ? get_option( 'mcs_confirmation_subject' ) : $defaults['mcs_confirmation_subject']; // subject line.
	$mcs_edit_confirmation_subject       = ( get_option( 'mcs_edit_confirmation_subject' ) ) ? get_option( 'mcs_edit_confirmation_subject' ) : $defaults['mcs_edit_confirmation_subject']; // subject line.
	$mcs_publication_subject             = ( get_option( 'mcs_publication_subject' ) ) ? get_option( 'mcs_publication_subject' ) : $defaults['mcs_publication_subject']; // subject line.
	$mcs_publication                     = ( get_option( 'mcs_publication' ) ) ? get_option( 'mcs_publication' ) : $defaults['mcs_publication']; // subject line.
	$mcs_dont_send_submitter_email       = get_option( 'mcs_dont_send_submitter_email' );
	$mcs_dont_send_admin_email           = get_option( 'mcs_dont_send_admin_email' );
	$mcs_html_email                      = get_option( 'mcs_html_email' );
	$edit_link                           = ( mcs_submit_url() ) ? ', <code>edit_link</code>' : '';
	$template_tags                       = mcs_notification_tags( $edit_link );

	$content = '<p>
				<input type="checkbox" id="mcs_dont_send_submitter_email" name="mcs_dont_send_submitter_email"' . checked( $mcs_dont_send_submitter_email, 'true', false ) . ' /> <label for="mcs_dont_send_submitter_email">' . __( 'Disable submitter email notification on automatically published events.', 'my-calendar-pro' ) . '</label>
			</p>
			<p>
				<input type="checkbox" id="mcs_dont_send_admin_email" name="mcs_dont_send_admin_email"' . checked( $mcs_dont_send_admin_email, 'true', false ) . ' /> <label for="mcs_dont_send_admin_email">' . __( 'Disable admin email notification on automatically published events.', 'my-calendar-pro' ) . '</label>
			</p>
			<p>
				<input type="checkbox" id="mcs_html_email" name="mcs_html_email"' . checked( $mcs_html_email, 'true', false ) . ' /> <label for="mcs_html_email">' . __( 'Send email notifications as HTML.', 'my-calendar-pro' ) . '</label>
			</p>
			<p>
				<label for="mcs_from">' . __( 'Send notifications from', 'my-calendar-pro' ) . '</label> <input type="text" name="mcs_from" id="mcs_from" class="widefat" value="' . esc_attr( $mcs_from ) . '" />
			</p>
			<ul class="toggles">
				<li id="tab_admin_notifications" class="active"><button type="button" href="#admin_notifications_tab" aria-controls="admin_notifications_tab" aria-expanded="true">' . __( 'Sent to Administrator', 'my-calendar-pro' ) . '</button></li>
				<li id="tab_submitter_notifications"><button type="button" href="#submitter_notifications_tab" aria-controls="submitter_notifications_tab" aria-expanded="false">' . __( 'Sent to Event Submitter', 'my-calendar-pro' ) . '</button></li>
			</ul>
			<fieldset class="notifications" id="admin_notifications_tab" aria-labelledby="tab_admin_notifications" aria-live="assertive">
				<legend class="screen-reader-text">' . __( 'Sent to administrators', 'my-calendar-pro' ) . '</legend>
				<p>
					<label for="mcs_to">' . __( 'Send notifications to:', 'my-calendar-pro' ) . '</label><br /><input type="text" name="mcs_to" id="mcs_to" class="widefat" value="' . esc_attr( $mcs_to ) . '" />
				</p>
				<h3>' . __( 'Email Templates', 'my-calendar-pro' ) . '</h3>
				' . $template_tags . '
				<div class="mc-notification-templates">
					<h4>' . __( 'New Events', 'my-calendar-pro' ) . '</h4>
					<p>
						<label for="mcs_subject">' . __( 'Notification Subject', 'my-calendar-pro' ) . '</label><br />
						<input type="text" name="mcs_subject" id="mcs_subject" class="widefat" value="' . stripslashes( esc_attr( $mcs_subject ) ) . '" />
					</p>
					<p>
						<label for="mcs_response">' . __( 'Notification message', 'my-calendar-pro' ) . '</label><br /><textarea class="widefat" name="mcs_response" id="mcs_response" rows="6" cols="60">' . stripslashes( esc_attr( $mcs_response ) ) . '</textarea>
					</p>
				</div>
				<div class="mc-notification-templates">
					<h4>' . __( 'Edited Events', 'my-calendar-pro' ) . '</h4>
					<p>
						<input type="checkbox"' . checked( $mcs_disable_edit_notifications, 'true', false ) . ' name="mcs_disable_edit_notifications" value="true" id="mcs_disable_edit_notifications" /> <label for="mcs_disable_edit_notifications">' . __( 'Disable Admin Notifications on Edits', 'my-calendar-pro' ) . '</label>
					</p>
					<p>
						<label for="mcs_edit_subject">' . __( 'Notification Subject (Edits)', 'my-calendar-pro' ) . '</label><br />
						<input type="text" name="mcs_edit_subject" id="mcs_edit_subject" class="widefat" value="' . stripslashes( esc_attr( $mcs_edit_subject ) ) . '" />
					</p>
					<p>
						<label for="mcs_edit_response">' . __( 'Notification message (Edits)', 'my-calendar-pro' ) . '</label><br /><textarea class="widefat" name="mcs_edit_response" id="mcs_edit_response" rows="6" cols="60">' . stripslashes( esc_attr( $mcs_edit_response ) ) . '</textarea>
					</p>
				</div>
			</fieldset>
			<fieldset class="notifications" id="submitter_notifications_tab" aria-labelledby="tab_submitter_notifications" aria-live="assertive">
				<legend class="screen-reader-text">' . __( 'Sent to event submitter', 'my-calendar-pro' ) . '</legend>
				<h3>' . __( 'Email Templates', 'my-calendar-pro' ) . '</h3>
				' . $template_tags . '
				<div class="mc-notification-templates">
					<h4>' . __( 'Event Submitted', 'my-calendar-pro' ) . '</h4>
					<p>
						<label for="mcs_confirmation_subject">' . __( 'Confirmation Subject', 'my-calendar-pro' ) . '</label><br />
						<input type="text" name="mcs_confirmation_subject" id="mcs_confirmation_subject" class="widefat" value="' . stripslashes( esc_attr( $mcs_confirmation_subject ) ) . '" />
					</p>
					<p>
						<label for="mcs_confirmation">' . __( 'Submitter confirmation message', 'my-calendar-pro' ) . '</label><br /><textarea class="widefat" name="mcs_confirmation" id="mcs_confirmation" rows="6" cols="60">' . stripslashes( esc_attr( $mcs_confirmation ) ) . '</textarea>
					</p>
				</div>
				<div class="mc-notification-templates">
					<h4>' . __( 'Event Edited', 'my-calendar-pro' ) . '</h4>
					<p>
						<input type="checkbox"' . checked( $mcs_disable_user_edit_notifications, 'true', false ) . ' name="mcs_disable_user_edit_notifications" value="true" id="mcs_disable_user_edit_notifications" /> <label for="mcs_disable_user_edit_notifications">' . __( 'Disable Submitter Notifications on Edits', 'my-calendar-pro' ) . '</label>
					</p>
					<p>
						<label for="mcs_edit_confirmation_subject">' . __( 'Confirmation Subject (Edits)', 'my-calendar-pro' ) . '</label><br />
						<input type="text" name="mcs_edit_confirmation_subject" id="mcs_edit_confirmation_subject" class="widefat" value="' . stripslashes( esc_attr( $mcs_edit_confirmation_subject ) ) . '" />
					</p>
					<p>
						<label for="mcs_edit_confirmation">' . __( 'Submitter confirmation message (Edits)', 'my-calendar-pro' ) . '</label><br /><textarea class="widefat" name="mcs_edit_confirmation" id="mcs_edit_confirmation" rows="6" cols="60">' . stripslashes( esc_attr( $mcs_edit_confirmation ) ) . '</textarea>
					</p>
				</div>
				<div class="mc-notification-templates">
					<h4>' . __( 'Event Published', 'my-calendar-pro' ) . '</h4>
					<p>
						<label for="mcs_publication_subject">' . __( 'Publication Subject', 'my-calendar-pro' ) . '</label><br />
						<input type="text" name="mcs_publication_subject" id="mcs_publication_subject" class="widefat" value="' . stripslashes( esc_attr( $mcs_publication_subject ) ) . '" />
					</p>
					<p>
						<label for="mcs_publication">' . __( 'Publication Message', 'my-calendar-pro' ) . '</label><br /><textarea class="widefat" name="mcs_publication" id="mcs_publication" rows="6" cols="60">' . stripslashes( esc_attr( $mcs_publication ) ) . '</textarea>
					</p>
				</div>
			</fieldset>';
	$label   = __( 'Update Notifications', 'my-calendar-pro' );
	$content = '<h2>' . __( 'Submission Notification Settings', 'my-calendar-pro' ) . '</h2>
		<div class="inside">' . $content . '{submit}</div>';

	$panels['notifications']['content'] = $content;
	$panels['notifications']['label']   = $label;

	return $panels;
}

/**
 * Notification template tags for display in settings.
 *
 * @param string $edit_link If edit link is available, include it.
 *
 * @return string
 */
function mcs_notification_tags( $edit_link ) {
	$fields = mcs_notification_fields();
	if ( $edit_link ) {
		array_push( $fields, 'edit_link' );
	}

	$output = '';
	foreach ( $fields as $field ) {
		$output .= '<code>{' . $field . '}</code>';
	}

	return '<p class="mcs-template-tags"><strong>' . __( 'Available tags:', 'my-calendar-pro' ) . '</strong> ' . $output . '</p>';
}

/**
 * Notification template placeholders.
 *
 * @param int|object $event Event ID or object used to populate details.
 *
 * @return array Array of template strings and the event object.
 */
function mcs_notification_fields( $event = false ) {
	if ( ! $event ) {
		$array = array( 'title', 'date', 'time', 'description', 'short', 'image', 'url', 'location', 'street', 'city', 'phone', 'blogname', 'fname', 'lname', 'name', 'email' );
	} else {
		if ( ! is_object( $event ) ) {
			$event_id = $event;
			$event    = mc_get_first_event( $event );
		} else {
			$event_id = $event->event_id;
		}
		$elabel  = '';
		$estreet = '';
		$ecity   = '';
		$ephone  = '';
		if ( property_exists( $event, 'location' ) && is_object( $event->location ) ) {
			$elabel  = $event->location->location_label;
			$estreet = $event->location->location_street;
			$ecity   = $event->location->location_city;
			$ephone  = $event->location->location_phone;
		}
		$array = array(
			'title'       => $event->event_title,
			'date'        => $event->event_begin,
			'time'        => $event->event_time,
			'description' => $event->event_desc,
			'short'       => $event->event_short,
			'image'       => $event->event_image,
			'url'         => $event->event_link,
			'location'    => $elabel,
			'street'      => $estreet,
			'city'        => $ecity,
			'phone'       => $ephone,
			'blogname'    => get_option( 'blogname' ),
			'edit_link'   => admin_url( "admin.php?page=my-calendar&mode=edit&event_id=$event_id" ),
			'event'       => $event,
		);
	}

	/**
	 * Filter the template tags used for calendar notifications after front-end submissions or edits.
	 *
	 * @hook mcs_notification_fields
	 *
	 * @param {array}  $array Array of keys and values used for notifications. Values omitted if no event passed.
	 * @param {object} $event Object used to build array.
	 *
	 * @return {array}
	 */
	return apply_filters( 'mcs_notification_fields', $array, $event );
}

/**
 * Send a notification submission to the admin.
 *
 * @param string $name Submitter name.
 * @param string $email Submitter email.
 * @param int    $event_id Event ID.
 * @param string $action Type of action.
 *
 * @return void
 */
function mcs_notify_admin( $name, $email, $event_id, $action ) {
	if ( preg_match( '/\s/', $name ) ) {
		list( $fname, $lname ) = preg_split( '/\s+(?=[^\s]+$)/', $name, 2 );
	} else {
		$fname = $name;
		$lname = '';
	}
	$array = mcs_notification_fields( $event_id );
	$event = $array['event'];
	unset( $array['event'] );

	$array['first_name'] = $fname;
	$array['last_name']  = $lname;
	$array['name']       = trim( $fname . ' ' . $lname );
	$array['email']      = $email;

	// if event is flagged as spam, don't send email notification.
	/**
	 * Filter to disable admin email notifications for custom reasons.
	 *
	 * @hook mcs_dont_send_admin_email
	 *
	 * @param {bool}  $no_email Default false to send an email.
	 * @param {int}   $event_id Event ID.
	 * @param {array} $array Array of information to include in email templates.
	 *
	 * @return {bool}
	 */
	$no_email = apply_filters( 'mcs_dont_send_admin_email', false, $event_id, $array );
	if ( 1 === (int) $event->event_approved && 'true' === get_option( 'mcs_dont_send_admin_email' ) ) {
		$no_email = true;
	}
	mcs_save_author( $fname, $lname, $email, $event );
	if ( 1 === (int) $event->event_flagged || true === $no_email ) {

		return;
	} else {
		// If configured to skip when editing, and action is an edit, move on silently.
		$skip_on_edit = get_option( 'mcs_disable_edit_notifications' );
		if ( $skip_on_edit && 'edit' === $action ) {
			return;
		}
		$subject      = ( '' === get_option( 'mcs_subject', '' ) ) ? 'New event on {blogname}' : get_option( 'mcs_subject' );
		$edit_subject = ( '' === get_option( 'mcs_edit_subject', '' ) ) ? 'Edited event on {blogname}' : get_option( 'mcs_edit_subject' );
		$message      = ( '' === get_option( 'mcs_response', '' ) ) ? 'New event from {first_name} {last_name}: {title}, {date}, {time}.' . PHP_EOL . PHP_EOL . 'Edit this event: {edit_link}' : get_option( 'mcs_response' );
		$edit_message = ( '' === get_option( 'mcs_edit_response', '' ) ) ? 'Edited event from {first_name} {last_name}: {title}, {date}, {time}.' . PHP_EOL . PHP_EOL . 'Edit again: {edit_link}' : get_option( 'mcs_edit_response' );

		if ( 'edit' === $action ) {
			$subject = $edit_subject;
			$message = $edit_message;
		}

		$subject = mc_draw_template( $array, $subject );
		$message = mc_draw_template( $array, $message );

		if ( 'true' === get_option( 'mcs_html_email' ) ) {
			add_filter( 'wp_mail_content_type', 'mcs_html_email' );
		}
		$mcs_to = ( '' === get_option( 'mcs_to', '' ) ) ? get_bloginfo( 'admin_email' ) : get_option( 'mcs_to' );
		if ( false !== strpos( $mcs_to, ',' ) ) {
			// remove any extra spaces.
			$mcs_to = array_map( 'trim', explode( ',', $mcs_to ) );
			$mcs_to = implode( ',', $mcs_to );
		}

		if ( is_email( $email ) ) {
			$headers = array( "From: \"$name\" <$email>" );
		} else {
			$headers = array( "From: \"$name\" <$mcs_to>" );
		}
		/**
		 * Filter admin notification email headers.
		 *
		 * @hook mcs_notify_admin_headers
		 *
		 * @param {array}  $headers Array of headers. Default includes 'From:' header only.
		 * @param {object} $event Event object.
		 *
		 * @return {array}
		 */
		$headers = apply_filters( 'mcs_notify_admin_headers', $headers, $event );
		$mail    = wp_mail( $mcs_to, $subject, $message, $headers );
		if ( ! $mail ) {
			// If mail fails, send without headers.
			wp_mail( $mcs_to, $subject, $message );
		}

		if ( 'true' === get_option( 'mcs_html_email' ) ) {
			remove_filter( 'wp_mail_content_type', 'mcs_html_email' );
		}
	}
}

/**
 * Send a notification submission to the event submitter.
 *
 * @param string $name Submitter name.
 * @param string $email Submitter email.
 * @param int    $event_id Event ID.
 * @param string $action Type of action.
 */
function mcs_notify_submitter( $name, $email, $event_id, $action ) {
	if ( preg_match( '/\s/', $name ) ) {
		list( $fname, $lname ) = preg_split( '/\s+(?=[^\s]+$)/', $name, 2 );
	} else {
		$fname = $name;
		$lname = '';
	}
	$array = mcs_notification_fields( $event_id );
	$event = $array['event'];
	unset( $array['event'] );

	$array['first_name'] = $fname;
	$array['last_name']  = $lname;
	$array['name']       = trim( $fname . ' ' . $lname );
	$array['email']      = $email;
	$array['edit_link']  = mcs_submit_url( $event_id, $event );

	// if event is flagged as spam, don't send email notification.
	/**
	 * Filter to disable submitter email notifications for custom reasons.
	 *
	 * @hook mcs_dont_send_submitter_email
	 *
	 * @param {bool}  $no_email Default false to send an email.
	 * @param {int}   $event_id Event ID.
	 * @param {array} $array Array of information to include in email templates.
	 *
	 * @return {bool}
	 */
	$no_email = apply_filters( 'mcs_dont_send_submitter_email', false, $event_id, $array );
	if ( 1 === (int) $event->event_approved && 'true' === get_option( 'mcs_dont_send_submitter_email' ) ) {
		$no_email = true;
	}
	if ( 1 === (int) $event->event_flagged || true === $no_email ) {
		return;
	} else {
		$subject      = get_option( 'mcs_confirmation_subject', 'New event on {blogname}' );
		$edit_subject = get_option( 'mcs_edit_confirmation_subject', 'Edited event on {blogname}' );
		$message      = get_option( 'mcs_confirmation', 'Thanks for proposing a new event, {first_name} {last_name}! {title}, {date}, {time}' );
		$edit_message = get_option( 'mcs_edit_confirmation', 'Thanks for proposing a new event, {first_name} {last_name}! {title}, {date}, {time}' );

		if ( 'edit' === $action ) {
			$subject = $edit_subject;
			$message = $edit_message;
		}

		$subject = mc_draw_template( $array, $subject );
		$message = mc_draw_template( $array, $message );
		$from    = ( '' === get_option( 'mcs_from', '' ) ) ? get_bloginfo( 'admin_email' ) : get_option( 'mcs_from' );
		if ( 'true' === get_option( 'mcs_html_email' ) ) {
			add_filter( 'wp_mail_content_type', 'mcs_html_email' );
		}

		$headers = array( "From: $from" );
		/**
		 * Filter submitter notification email headers.
		 *
		 * @hook mcs_notify_submitter_headers
		 *
		 * @param {array}  $headers Array of headers. Default includes 'From:' header only.
		 * @param {object} $event Event object.
		 *
		 * @return {array}
		 */
		$headers = apply_filters( 'mcs_notify_submitter_headers', $headers, $event );
		$mail    = wp_mail( $email, $subject, $message, $headers );
		if ( ! $mail ) {
			// If mail fails, send without headers.
			wp_mail( $email, $subject, $message );
		}

		if ( 'true' === get_option( 'mcs_html_email' ) ) {
			remove_filter( 'wp_mail_content_type', 'mcs_html_email' );
		}
	}
}

/**
 * Send a notification submission to the event submitter if an admin publishes the post.
 *
 * @param int    $prev_status Previous status.
 * @param int    $new_status (1 = published).
 * @param string $action Action performed.
 * @param array  $data Event array. Not used.
 * @param int    $event_id Event ID.
 *
 * @return void;
 */
function mcs_notify_submitter_transition( $prev_status, $new_status, $action, $data, $event_id ) {
	// If the previous status was any unpublished status and becomes published.
	$published = ( 1 !== (int) $prev_status && 1 === (int) $new_status && 'edit' === $action ) ? true : false;

	// If configured to skip when editing, and action is an edit other than publishing, move on silently.
	$skip_on_edit = get_option( 'mcs_disable_user_edit_notifications' );
	if ( $skip_on_edit && 'edit' === $action && ! $published ) {
		return;
	}
	// If this message has just been sent, ignore.
	$sent = get_transient( 'mcs_notification_sent' );
	if ( (int) $sent === (int) $event_id ) {
		return;
	}

	$post_ID   = mc_get_event_post( $event_id );
	$submitter = get_post_meta( $post_ID, '_submitter_details', true );
	$fname     = '';
	$lname     = '';
	$email     = '';
	if ( is_array( $submitter ) && ! empty( $submitter ) ) {
		$fname = $submitter['first_name'];
		$lname = $submitter['last_name'];
		$email = $submitter['email'];
	}
	if ( ! is_email( $email ) ) {
		// If email address is missing or invalid, this message can't be sent.
		return;
	}
	$array = mcs_notification_fields( $event_id );
	$event = $array['event'];
	unset( $array['event'] );

	$array['first_name'] = $fname;
	$array['last_name']  = $lname;
	$array['name']       = trim( $fname . ' ' . $lname );
	$array['email']      = $email;
	$array['edit_link']  = mcs_submit_url( $event_id, $event );

	/**
	 * Filter allows you to disable email notifications for various custom reasons.
	 *
	 * @hook mcs_dont_send_submitter_email
	 *
	 * @param bool  $no_email Default false. Return true to skip submitter email.
	 * @param int   $event_id Event ID.
	 * @param array $array Data array to use for email templates.
	 *
	 * @return bool
	 */
	$no_email = apply_filters( 'mcs_dont_send_submitter_email', false, $event_id, $array );
	if ( 'true' === get_option( 'mcs_dont_send_submitter_email' ) || $no_email ) {
		return;
	}

	if ( $published ) {
		$edit_subject = get_option( 'mcs_publication_subject', 'Event published on {blogname}' );
		$edit_message = get_option( 'mcs_publication', 'Your event has been published, {first_name} {last_name}! {title}, {date}, {time}' );
	} else {
		$edit_subject = get_option( 'mcs_edit_subject', 'Event on {blogname} has been edited' );
		$edit_message = get_option( 'mcs_edit_confirmation', 'Your event has been edited, {first_name} {last_name}! {title}, {date}, {time}' );
	}

	$subject = mc_draw_template( $array, $edit_subject );
	$message = mc_draw_template( $array, $edit_message );
	$from    = ( '' === get_option( 'mcs_from', '' ) ) ? get_bloginfo( 'admin_email' ) : get_option( 'mcs_from' );
	if ( 'true' === get_option( 'mcs_html_email' ) ) {
		add_filter( 'wp_mail_content_type', 'mcs_html_email' );
	}

	$headers = array( "From: $from" );
	/**
	 * Filter submitter notification email headers.
	 *
	 * @hook mcs_notify_submitter_headers
	 *
	 * @param {array}  $headers Array of headers. Default includes 'From:' header only.
	 * @param {object} $event Event object.
	 *
	 * @return {array}
	 */
	$headers = apply_filters( 'mcs_notify_submitter_headers', $headers, $event );
	$mail    = wp_mail( $email, $subject, $message, $headers );
	if ( ! $mail ) {
		// If mail fails, send without headers.
		wp_mail( $email, $subject, $message );
	}
	// Set transient to avoid duplicating this message.
	set_transient( 'mcs_notification_sent', $event_id, 15 );

	if ( 'true' === get_option( 'mcs_html_email' ) ) {
		remove_filter( 'wp_mail_content_type', 'mcs_html_email' );
	}
}

// Use Codemirror for email fields when enabled.
add_action(
	'admin_enqueue_scripts',
	function () {
		if ( ! function_exists( 'wp_enqueue_code_editor' ) ) {
			return;
		}
		if ( 'my-calendar_page_my-calendar-submissions' !== get_current_screen()->id ) {
			return;
		}

		// Enqueue code editor and settings for manipulating HTML.
		$settings = wp_enqueue_code_editor(
			array(
				'type'       => 'text/html',
				'codemirror' => array(
					'autoRefresh' => true,
				),
			)
		);

		// Bail if user disabled CodeMirror or using default styles.
		if ( false === $settings || 'true' !== get_option( 'mcs_html_email' ) ) {
			return;
		}
		wp_add_inline_script(
			'code-editor',
			sprintf(
				'jQuery( function() { wp.codeEditor.initialize( "mcs_response", %s ); } );',
				wp_json_encode( $settings )
			)
		);
		wp_add_inline_script(
			'code-editor',
			sprintf(
				'jQuery( function() { wp.codeEditor.initialize( "mcs_edit_response", %s ); } );',
				wp_json_encode( $settings )
			)
		);
		wp_add_inline_script(
			'code-editor',
			sprintf(
				'jQuery( function() { wp.codeEditor.initialize( "mcs_confirmation", %s ); } );',
				wp_json_encode( $settings )
			)
		);
		wp_add_inline_script(
			'code-editor',
			sprintf(
				'jQuery( function() { wp.codeEditor.initialize( "mcs_edit_confirmation", %s ); } );',
				wp_json_encode( $settings )
			)
		);
		wp_add_inline_script(
			'code-editor',
			sprintf(
				'jQuery( function() { wp.codeEditor.initialize( "mcs_publication", %s ); } );',
				wp_json_encode( $settings )
			)
		);
	}
);
