<?php
/**
 * REST API Client Post process.
 *
 * @category API
 * @package  My Calendar Pro
 * @author   Joe Dolson
 * @license  GPLv3
 * @link     https://www.joedolson.com/my-calendar-pro/
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Get registered API endpoints.
 *
 * List of sites the current site is able to post to.
 *
 * @return array supported endpoints.
 */
function mcs_api_endpoints() {
	// Currently, all connections have access to add, edit, and delete.
	$settings  = ( is_array( get_option( 'mc_api_client_settings' ) ) ) ? get_option( 'mc_api_client_settings' ) : array();
	$settings  = array_merge(
		array(
			'mc_api_key'       => '',
			'mc_api_endpoints' => array(),
		),
		$settings
	);
	$endpoints = isset( $settings['mc_api_endpoints'] ) ? $settings['mc_api_endpoints'] : array();
	$endpoints = apply_filters( 'my_calendar_api_endpoints', $endpoints );

	// remove current site from list of endpoints.
	// currently doesn't verify protocol match; do I want to restrict to SSL only.
	if ( isset( $endpoints[ home_url() ] ) ) {
		unset( $endpoints[ home_url() ] );
	}

	return $endpoints;
}

/**
 * Get a specific endpoints info from URL provided.
 *
 * @param string $endpoint URL.
 *
 * @return array|boolean
 */
function mcs_get_endpoint( $endpoint ) {
	$endpoints = mcs_api_endpoints();
	foreach ( $endpoints as $ep => $data ) {
		if ( $ep === $endpoint ) {
			return $data;
		}
	}

	return false;
}

/**
 * Get the key for a specific endpoint. Empty string if not found.
 *
 * @param string $endpoint URL.
 *
 * @return string
 */
function mcs_get_api_key( $endpoint ) {
	$endpoint = mcs_get_endpoint( $endpoint );
	$key      = '';

	if ( $endpoint ) {
		$key = $endpoint['key'];
	}

	return $key;
}

/**
 * After event saved successfully, post to api endpoints.
 *
 * @param string $action Action sent from My Calendar.
 * @param array  $data Data from My Calendar.
 * @param int    $event_id ID of event in source site.
 * @param string $result Message response from My Calendar insert.
 *
 * @return void
 */
function mcs_api_post( $action, $data, $event_id, $result ) {
	$sites     = mcs_api_endpoints();
	$responses = array();
	// post the raw data.
	$data = map_deep( $_POST, 'sanitize_textarea_field' );

	if ( empty( $sites ) ) {
		return;
	} else {
		foreach ( $sites as $endpoint => $site ) {
			if ( isset( $_POST['mc_api_endpoints'] ) && in_array( $endpoint, $_POST['mc_api_endpoints'], true ) ) {
				$author   = $site['author'];
				$verified = mcs_api_verify_endpoint( $endpoint );
				$key      = mcs_get_api_key( $endpoint );
				if ( ! $verified ) {
					$responses[ $endpoint ] = $verified;
				} else {
					// modify author to endpoint defined author.
					$data['event_author'] = $author;
					// send query to endpoint.
					$endpoint = $endpoint; // need to add arguments to endpoint? Simpler if I pass action as POST.
					$response = mcs_api_post_event( $data, $endpoint, $key, $event_id, $action );
					if ( is_wp_error( $response ) ) {
						$event_post             = mc_get_event_post( $event_id );
						$responses[ $endpoint ] = array(
							'body'     => array(
								'message'  => $response->get_error_message(),
								'event_id' => $event_post,
							),
							'endpoint' => $endpoint,
						);
					} else {
						$responses[ $endpoint ] = array(
							'body'     => $response['body'],
							'endpoint' => $endpoint,
						);
					}
				}
			}
		}
	}

	// if no data was generated, don't log.
	if ( ! empty( $responses ) ) {
		mcs_api_log_responses( $responses, $event_id );
	}
}
add_action( 'mc_save_event', 'mcs_api_post', 20, 4 );

/**
 * Delete an event from another site.
 *
 * @param int $event_id Event ID.
 * @param int $post_id Post ID.
 *
 * @return void
 */
function mcs_api_delete( $event_id, $post_id ) {
	$sites     = mcs_api_endpoints();
	$responses = array();
	// post the raw data.
	$data = map_deep( $_POST, 'sanitize_textarea_field' );

	if ( empty( $sites ) ) {
		return;
	} else {
		foreach ( $sites as $endpoint => $site ) {
			$verified = mcs_api_verify_endpoint( $endpoint );
			if ( ! $verified ) {
				$responses[ $endpoint ] = $verified;
			} else {
				// send query to endpoint.
				mcs_api_delete_event( $data, $endpoint, $event_id );
			}
		}
	}
}
add_action( 'mc_delete_event', 'mcs_api_delete', 10, 2 );

/**
 * Bulk delete of events.
 *
 * @param array $ids Array of event IDs.
 */
function mcs_api_mass_delete( $ids ) {
	if ( is_array( $ids ) ) {
		foreach ( $ids as $id ) {
			mcs_api_delete( $id, false );
		}
	}
}
add_action( 'mc_mass_delete_events', 'mcs_api_mass_delete', 10, 1 );

/**
 * Delete an event.
 *
 * @param array  $data Event data.
 * @param string $endpoint Target API to send request to.
 * @param int    $event_id Event ID.
 */
function mcs_api_delete_event( $data, $endpoint, $event_id ) {
	$endpoint = trailingslashit( $endpoint ) . 'wp-json/mc_api/v1/events/' . $event_id;

	$args = array(
		'body'   => array(
			'data'            => $data,
			'event_id'        => $event_id,
			'source'          => home_url(),
			'action'          => 'delete',
			'mc_api_auth_key' => mcs_api_client_authentication_key(),
		),
		'method' => 'DELETE',
	);

	// wp_safe_remote_posts automatically rejects URLs with excess redirects or otherwise not trusted.
	wp_safe_remote_post( $endpoint, $args );
}

/**
 * Given a URL, verify that the endpoint is a valid URL.
 *
 * No reason to ping it; if it's a 404, wp_remote_post will do that.
 *
 * @param string $url Endpoint URL.
 *
 * @return string $url.
 */
function mcs_api_verify_endpoint( $url ) {
	// return valid URL or falsey value.
	$url = esc_url( $url );

	return $url;
}

/**
 * Do post action
 *
 * @param array  $data POST data formatted for My Calendar.
 * @param string $endpoint URL to submit data to.
 * @param string $key Authentication Key.
 * @param int    $event_id Event ID from original site for new, event ID source ID for edited events.
 * @param string $action Add/Update/Delete.
 *
 * @return HTTP response array
 */
function mcs_api_post_event( $data, $endpoint, $key, $event_id, $action ) {
	// post data, submit original event ID and home_url as source parameters in new event.
	$categories = mc_get_categories( $event_id );
	$cats       = array();
	foreach ( $categories as $cat ) {
		$cats[] = mc_get_category_detail( $cat, false );
	}

	if ( 'add' === $action ) {
		$endpoint = trailingslashit( $endpoint ) . 'wp-json/mc_api/v1/events';
	}

	if ( 'edit' === $action || 'delete' === $action ) {
		$endpoint = trailingslashit( $endpoint ) . 'wp-json/mc_api/v1/events/' . $event_id;
	}

	$args = array(
		'body' => array(
			'data'            => $data,
			'category'        => $cats,
			'event_id'        => $event_id,
			'source'          => home_url(),
			'action'          => $action,
			'mc_api_auth_key' => ( empty( $key ) ) ? mcs_api_client_authentication_key() : $key,
		),
	);

	// wp_safe_remote_posts automatically rejects URLs with excess redirects or otherwise not trusted.
	$response = wp_safe_remote_post( $endpoint, $args );

	return $response;
}

/**
 * Fetches authentication key sent in query
 *
 * @return return authentication key
 */
function mcs_api_client_authentication_key() {
	$settings   = ( is_array( get_option( 'mc_api_client_settings' ) ) ) ? get_option( 'mc_api_client_settings' ) : array();
	$settings   = array_merge(
		array(
			'mc_api_key'       => '',
			'mc_api_endpoints' => array(),
		),
		$settings
	);
	$mc_api_key = $settings['mc_api_key'];

	return apply_filters( 'mc_api_authentication_key', $mc_api_key );
}
