<?php
/**
 * Responsive mode
 *
 * @category Features
 * @package  My Calendar Pro
 * @author   Joe Dolson
 * @license  GPLv3
 * @link     https://www.joedolson.com/my-calendar-pro/
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

add_filter( 'mcs_custom_settings_update', 'mcs_responsive_update', 10, 2 );
/**
 * Update settings for responsive options.
 *
 * @param mixed $value Value.
 * @param array $post POST data.
 *
 * @return string
 */
function mcs_responsive_update( $value, $post ) {
	// save settings.
	if ( isset( $post['responsive_settings'] ) ) {
		if ( isset( $post['mcs_responsive_mode'] ) ) {
			update_option( 'mcs_responsive_mode', 'true' );
		} else {
			delete_option( 'mcs_responsive_mode' );
		}
		$stylesheet           = ( isset( $post['mcs_stylesheet'] ) ) ? sanitize_text_field( $post['mcs_stylesheet'] ) : '';
		$mcs_stylesheet       = ( in_array( $stylesheet, mcs_responsive_stylesheets(), true ) ) ? $stylesheet : 'twentyfifteen';
		$mcs_responsive_setup = ( isset( $post['mcs_responsive_setup'] ) && '1' === $post['mcs_responsive_setup'] ) ? '1' : '2';
		update_option( 'mcs_stylesheet', $mcs_stylesheet );
		update_option( 'mcs_responsive_setup', $mcs_responsive_setup );

		return __( 'Responsive Mode Updated', 'my-calendar-pro' );
	}

	return $value;
}

/**
 * Return array of valid responsive stylesheets.
 *
 * @return array
 */
function mcs_responsive_stylesheets() {
	$sheets = array(
		'TwentyFifteen' => 'twentyfifteen',
		'Blue'          => 'blue',
		'Green'         => 'green',
		'Light'         => 'light',
		'Dark'          => 'dark',
		'Basic'         => 'basic',
	);

	return apply_filters( 'mcs_responsive_stylesheets', $sheets );
}

add_filter( 'mcs_settings_tabs', 'mcs_responsive_tabs' );
/**
 * Add responsive tab to tab list.
 *
 * @param array $tabs existing tabs.
 *
 * @return array $tabs
 */
function mcs_responsive_tabs( $tabs ) {
	$enabled = get_option( 'mcs_responsive_mode' );
	if ( 'true' === $enabled ) {
		$tabs['responsive'] = __( 'Responsive', 'my-calendar-pro' );
	}

	return $tabs;
}

add_filter( 'mcs_settings_panels', 'mcs_responsive_settings' );
/**
 * Add responsive panel to panel list.
 *
 * @param array $panels existing panels.
 *
 * @return array $panels
 */
function mcs_responsive_settings( $panels ) {
	$enabled = get_option( 'mcs_responsive_mode' );
	if ( 'true' !== $enabled ) {
		return $panels;
	}
	$mcs_responsive_mode  = get_option( 'mcs_responsive_mode' );
	$mcs_stylesheet       = get_option( 'mcs_stylesheet' );
	$mcs_responsive_setup = ( '' !== get_option( 'mcs_responsive_setup', '' ) ) ? get_option( 'mcs_responsive_setup' ) : 1;

	if ( 'true' === $mcs_responsive_mode ) {

		$sheets = '';
		foreach ( mcs_responsive_stylesheets() as $name => $key ) {
			$sheets .= "<option value='$key'" . selected( $mcs_stylesheet, $key, false ) . '>' . $name . '</option>';
		}

		$disabled = ( 1 === (int) $mcs_responsive_setup ) ? ' disabled="disabled"' : '';

		$controls = '<fieldset>
			<legend>' . __( 'Responsive Stylesheet', 'my-calendar-pro' ) . "</legend>
			<p id='responsive_styles_note'>
				" . __( 'Responsive mode uses different HTML than the default My Calendar stylesheets. Default stylesheets may not render correctly when responsive mode is enabled.', 'my-calendar-pro' ) . "
			</p>
			<p>
				<input type='radio' name='mcs_responsive_setup' id='mcs_basic_only' value='1'" . checked( $mcs_responsive_setup, '1', false ) . " /> <label for='mcs_basic_only'>" . __( 'Enable base responsive CSS to supplement default styles', 'my-calendar-pro' ) . "</label><br />
				<input type='radio' name='mcs_responsive_setup' id='mcs_replacement' value='2'" . checked( $mcs_responsive_setup, '2', false ) . " /> <label for='mcs_replacement'>" . __( 'Replace My Calendar default styles', 'my-calendar-pro' ) . "</label>
			</p>
			<p>
				<label for='mcs_stylesheet'>" . __( 'Select Responsive Theme', 'my-calendar-pro' ) . "</label>
				<select name='mcs_stylesheet' id='mcs_stylesheet'$disabled>
					<option value='default'>" . __( 'No styles', 'my-calendar-pro' ) . "</option>
					$sheets
				</select>
			</p>
		</fieldset>";
	} else {
		$controls = "<div>
			<input type='hidden' name='mcs_content' value='" . esc_attr( $mcs_stylesheet ) . "' />
		</div>";
	}
	$panels['responsive'] = '<h2>' . __( 'Responsive Mode', 'my-calendar-pro' ) . '</h2>
		<div class="inside">
			<p>' . __( 'Since My Calendar Pro 2.0.0, all My Calendar styles are automatically made responsive with Pro installed. It is only necessary to enable responsive mode if you wish to also change the underlying HTML semantics.', 'my-calendar-pro' ) . '</p>
			<p>
				<input type="checkbox" name="mcs_responsive_mode" id="mcs_responsive_mode" value="true" ' . checked( $mcs_responsive_mode, 'true', false ) . '/> <label for="mcs_responsive_mode">' . __( 'Enable Responsive Mode', 'my-calendar-pro' ) . '</label>
			</p>' . $controls . '
			{submit}
		</div>';

	return $panels;
}

add_filter( 'mc_format_toggle_html', 'mcs_disable_toggle', 10, 3 );
/**
 * In responsive mode, no toggle between grid and list mode; they're the same.]
 *
 * @param string $toggle Toggle HTML.
 * @param string $format Viewed format.
 * @param string $time Viewed time.
 *
 * @return string
 */
function mcs_disable_toggle( $toggle, $format, $time ) {
	$enabled = get_option( 'mcs_responsive_mode' );
	if ( 'true' === $enabled ) {
		$toggle = '';
	}

	return $toggle;
}

add_filter( 'mc_registered_stylesheet', 'mcs_responsive_styles' );
/**
 * Get registered stylesheet path.
 *
 * @param string $stylesheet Name of stylesheet.
 *
 * @return string.
 */
function mcs_responsive_styles( $stylesheet ) {
	$enabled = get_option( 'mcs_responsive_mode' );
	$setup   = get_option( 'mcs_responsive_setup' );
	if ( '2' === (string) $setup ) {
		$mcs_stylesheet = get_option( 'mcs_stylesheet' );
		if ( 'true' === $enabled ) {
			if ( file_exists( get_stylesheet_directory() . '/mc-responsive.css' ) ) {
				$stylesheet = esc_url( get_stylesheet_directory() . '/mc-responsive.css' );
			} else {
				if ( '' !== $mcs_stylesheet ) {
					$mcs_stylesheet = mcs_get_file( 'styles/' . $mcs_stylesheet . '.css', 'url' );
					$stylesheet     = esc_url( $mcs_stylesheet );
				}
			}
		}
	}

	return $stylesheet;
}

add_action( 'wp_enqueue_scripts', 'mcs_responsive' );
/**
 * Enqueue custom styles for responsive.
 */
function mcs_responsive() {
	global $mcs_version;
	$enabled = get_option( 'mcs_responsive_mode' );
	if ( '1' === get_option( 'mcs_responsive_setup' ) && 'true' === $enabled ) {
		wp_enqueue_style( 'mcs.base', mcs_get_file( 'styles/base.css', 'url' ), array( 'my-calendar-style' ), $mcs_version );
	}
}

/**
 * Fetch file required by My Calendar. Checks stylesheet directory and calendar custom directory in order.
 *
 * @param string $file File name.
 * @param string $type Path or URL.
 *
 * @return string.
 */
function mcs_get_file( $file, $type = 'path' ) {
	$dir  = plugin_dir_path( __FILE__ );
	$url  = plugin_dir_url( __FILE__ );
	$base = basename( $dir );
	$path = ( 'path' === $type ) ? $dir . $file : $url . $file;
	if ( file_exists( get_stylesheet_directory() . '/' . $file ) ) {
		$path = ( 'path' === $type ) ? get_stylesheet_directory() . '/' . $file : get_stylesheet_directory_uri() . '/' . $file;
	}
	if ( file_exists( str_replace( $base, 'my-calendar-custom', $dir ) . $file ) ) {
		$path = ( 'path' === $type ) ? str_replace( $base, 'my-calendar-custom', $dir ) . $file : str_replace( $base, 'my-calendar-custom', $url ) . $file;
	}
	$path = apply_filters( 'mc_get_file', $path, $file );

	return $path;
}

add_action( 'init', 'mcs_activate_responsive_mode' );
/**
 * Enable responsive mode.
 */
function mcs_activate_responsive_mode() {
	$enabled = get_option( 'mcs_responsive_mode' );
	if ( 'true' === $enabled ) {
		add_filter( 'mc_grid_wrapper', 'mcs_grid_wrapper', 10, 2 );
		add_filter( 'mc_grid_week_wrapper', 'mcs_week_wrapper', 10, 2 );
		add_filter( 'mc_grid_header_wrapper', 'mcs_header_wrapper', 10, 2 );
		add_filter( 'mc_grid_day_wrapper', 'mcs_day_wrapper', 10, 2 );
		add_filter( 'mc_grid_caption', 'mcs_grid_caption', 10, 2 );
	}
}

/**
 * Filter heading element for responsive.
 *
 * @param string $element Default element.
 * @param string $format Format viewed.
 *
 * @return string
 */
function mcs_grid_caption( $element, $format ) {
	return ( 'calendar' === $format ) ? 'h2' : $element;
}

/**
 * Filter grid wrapper element for responsive.
 *
 * @param string $element Default element.
 * @param string $format Format viewed.
 *
 * @return string
 */
function mcs_grid_wrapper( $element, $format ) {
	return ( 'calendar' === $format ) ? 'div' : $element;
}

/**
 * Filter week wrapper element for responsive.
 *
 * @param string $element Default element.
 * @param string $format Format viewed.
 *
 * @return string
 */
function mcs_week_wrapper( $element, $format ) {
	return ( 'calendar' === $format ) ? 'ul' : $element;
}

/**
 * Filter header wrapper element for responsive.
 *
 * @param string $element Default element.
 * @param string $format Format viewed.
 *
 * @return string
 */
function mcs_header_wrapper( $element, $format ) {
	return ( 'calendar' === $format ) ? 'li' : $element;
}

/**
 * Filter day wrapper element for responsive.
 *
 * @param string $element Default element.
 * @param string $format Format viewed.
 *
 * @return string
 */
function mcs_day_wrapper( $element, $format ) {
	return ( 'calendar' === $format ) ? 'li' : $element;
}

add_action( 'mc_print_view_head', 'mcs_responsive_print_styles' );
/**
 * Add responsive styles to print view.
 */
function mcs_responsive_print_styles() {
	$enabled = get_option( 'mcs_responsive_mode' );
	if ( 'true' === $enabled ) {
		$url = plugin_dir_url( __FILE__ );
		if ( mc_file_exists( 'css/mcs-print.css' ) ) {
			$stylesheet = mc_get_file( 'css/mcs-print.css', 'url' );
		} else {
			$stylesheet = $url . 'css/mcs-print.css';
		}

		echo '<link rel="stylesheet" href="' . esc_url( $stylesheet ) . '" type="text/css" media="screen,print" />';
	}
}
